program station_indices

! version 3.3.1

! Calculate extreme indices from station data in a single format
! Format is:

!      lat,long  (-ve latitude for southern hemisphere, longitude format not important)
! yr,mth,day,tmin,tmax,tmean,precip
!      ...
!      ...
!      ...
! yr,mth,day,tmin,tmax,tmean,precip

! fields need to be separated by a comma or space

! Stations entered as arguments to program or, if no arguments, names are assumed to be in file station_list.dat

include 'indices.inc'

! top 10 indices
integer ntop	
parameter(ntop=10)

integer yr,mth,day,ivar,lu,linenum,iargc,ifile,i,lu1,id,lngstr,startyr,endyr,itop(ntop)
character infile*120,outfile*120,fmt*30,listfile*30,line*10000
logical uselist,hasdata(nIndices,5),top10,wantIndex(nIndices)
real tmin,tmax,tmean,precip,a,r,tau,z,lat,lon
real lintrend(nIndices,5),trendsig(nIndices,5),x(minyr:maxyr),y(minyr:maxyr)
character*3 ssns(5)

data ssns/'DJF','MAM','JJA','SON','ANN'/
data itop/8,9,11,17,31,53,55,41,56,57/	! STARDEX top 10 indices

!!!!!!! User Parameters !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

bm_minyr=1961        ! start year of base period for normals
bm_maxyr=1990        ! end year of base period for normals
miss=-999.9          ! missing value of input data
wd_cutoff=1.0        ! minimum rain for wet day classification
frac_thresh=0.8      ! minimum fraction of non-missing data allowed
DegDays_Thresh=5.0   ! threshold for degree day and growing season calculations
minPercentileDays=10 ! minimum no. of days for which we can calculate a percentile 
newnames=.true.     ! use new (more consistent) names 
top10=.false.         ! output STARDEX top 10 indices only

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

listfile='station_list.dat'
lu=21
lu1=22
ifile=0

! flag desired indices
if(top10)then
 do i=1,nIndices
  wantIndex(i)=.false.
 end do
 do i=1,ntop
  wantIndex(itop(i))=.true.
 end do
else
 do i=1,nIndices
  wantIndex(i)=.true.
 end do
end if

if(iargc().lt.1)then
 uselist=.true.
 open(lu1,file=listfile,action='READ',status='old',err=991)
else
 uselist=.false.
end if

! loop over stations
30 continue

 ! inits
 do yr=minyr,maxyr
  do mth=1,12
   do day=1,31
    do ivar=1,4
     data_in(yr,mth,day,ivar)=miss
    end do
   end do
  end do
 end do
 linenum=0
 ifile=ifile+1

 ! get station name and open file
 if(uselist)then
  read(lu1,'(a)',end=40,err=993)infile
 else
  if(ifile.gt.iargc())goto 40
  call getarg(ifile,infile)
 end if
 open(lu,file=infile,action='READ',status='old',err=992)

 ! read file
 read(lu,*)lat,lon
 if(lat.lt.-90.or.lat.gt.90)stop 'error: latitude must be between -90 and 90'
 if(lat.gt.0)then
  south=.false.
 else
  south=.true.
 end if
 10 linenum=linenum+1
  read(lu,*,err=990,end=20)yr,mth,day,tmin,tmax,tmean,precip
 if(yr.ge.minyr.and.yr.le.maxyr)then
  data_in(yr,mth,day,1)=precip
  data_in(yr,mth,day,2)=tmax
  data_in(yr,mth,day,3)=tmin
  data_in(yr,mth,day,4)=tmean
 end if
 goto 10
 20 continue
 close(lu)

 ! indices
 call extremes_indices()

 ! linear trend of indices with kendall-tau significance test
 do i=1,nIndices
  do mth=1,5
   do yr=minyr,maxyr
    x(yr)=yr
    y(yr)=ts(i,yr,mth)
   end do
   call leastsq(x,y,nyrs,miss,a,lintrend(i,mth),r)
   call kendall_miss(x,y,nyrs,tau,z,trendsig(i,mth),miss)
  end do
 end do

 ! output (series with data only)

 do i=1,nIndices
  do mth=1,5
   hasdata(i,mth)=.false.
   if(wantIndex(i))then
    do yr=minyr,maxyr
     if(ts(i,yr,mth).ne.miss)hasdata(i,mth)=.true.
    end do
   end if
  end do
 end do

 startyr=0
 endyr=-1
 do yr=minyr,maxyr
  do i=1,nIndices
   do mth=1,5
    if(ts(i,yr,mth).ne.miss.and.wantIndex(i))then
     if(startyr.eq.0)startyr=yr
     endyr=yr
    end if
   end do
  end do
 end do
 
 outfile=trim(infile)//'.ind.csv'
 open(lu,file=outfile)

 line='Year'
 i=5
 do id=1,nIndices
  do mth=1,5
   if(hasdata(id,mth))then
    line(i:)=','//names(id)
    i=i+1+lngstr(names(id))
   end if
  end do
 end do
 write(lu,'(a)')trim(line)

 i=1
 do id=1,nIndices
  do mth=1,5
   if(hasdata(id,mth))then
    line(i:)=','//ssns(mth)
    i=i+1+lngstr(ssns(mth))
   end if
  end do
 end do
 write(lu,'(a)')trim(line)

 do yr=startyr,endyr
  write(line,'(i4)')yr
  i=5
  do id=1,nIndices
   do mth=1,5
    if(hasdata(id,mth))then
     line(i:)=','
     i=i+1
     if(ts(id,yr,mth).ne.miss)then
      write(line(i:),'(f10.4)')ts(id,yr,mth)
      i=i+10
     end if
    end if
   end do
  end do
  write(lu,'(a)')trim(line)
 end do

 line='trend'
 i=6
 do id=1,nIndices
  do mth=1,5
   if(hasdata(id,mth))then
    line(i:)=','
    i=i+1
    if(lintrend(id,mth).ne.miss)then
     write(line(i:),'(f10.4)')lintrend(id,mth)
     i=i+10
    end if
   end if
  end do
 end do
 write(lu,'(a)')trim(line)

 line='p <'
 i=4
 do id=1,nIndices
  do mth=1,5
   if(hasdata(id,mth))then
    line(i:)=','
    i=i+1
    if(trendsig(id,mth).ne.miss)then
     write(line(i:),'(f10.4)')trendsig(id,mth)
     i=i+10
    end if
   end if
  end do
 end do
 write(lu,'(a)')trim(line)

 close(lu)

goto 30
40 continue

if(uselist)close(lu1)

stop

990 write(0,'(3a,i8)')'Read error in file ',trim(infile),' at line ',linenum
stop

991 write(0,'(a)')'Usage: station_indices file1 [file2...]'
write(0,'(a)')'Omit files when filenames in station_list.dat'
stop

992 write(0,'(2a)')'Error opening file: ',trim(infile)
stop

993 write(0,'(a)')'Error reading file station_list.dat'
stop

end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

subroutine kendall_miss(x,y,n,tau,z,prob,miss)
! entry point for missing data

implicit none

integer n,i,nn
real x(n),y(n),tau,z,prob,miss,data1(n),data2(n)

nn=0
do i=1,n
 if(x(i).ne.miss.and.y(i).ne.miss)then
  nn=nn+1
  data1(nn)=x(i)
  data2(nn)=y(i)
 end if
end do

if(nn.gt.2)then
 call kendall(data1,data2,nn,tau,z,prob)
else
 tau=miss
 z=miss
 prob=miss
end if

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

subroutine kendall(data1,data2,n,tau,z,prob)
! tau: Kendall's Tau
! z: number of std devs from 0 of tau
! prob: tau's probability

implicit none

integer n,n1,n2,is,j,k
real data1(n),data2(n),a1,a2,aa,tau,var,prob,z,erfcc

n1=0
n2=0
is=0

do j=1,n-1
 do k=j+1,n
  a1=data1(j)-data1(k)
  a2=data2(j)-data2(k)
  aa=a1*a2
  if(aa.ne.0.0)then
   n1=n1+1
   n2=n2+1
   if(aa.gt.0.0)then
    is=is+1
   else
    is=is-1
   end if
  else
   if(a1.ne.0.0)n1=n1+1
   if(a2.ne.0.0)n2=n2+1
  end if
 end do
end do

if(n1*n2.ne.0)then
 tau=float(is)/sqrt(float(n1)*float(n2))
 var=(4.0*n+10.0)/(9.0*n*(n-1.0))
 z=tau/sqrt(var)
 prob=erfcc(abs(z)/1.4142136)
else
 tau=0
 z=0
 prob=1
end if

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

real function erfcc(x)

implicit none

real x,z,t

z=abs(x)
t=1.0/(1.0+0.5*z)
erfcc=t*exp(-z*z-1.26551223+t*(1.00002368+t*(.37409196+ &
 t*(.09678418+t*(-.18628806+t*(.27886807+t*(-1.13520398+ &
 t*(1.48851587+t*(-.82215223+t*.17087277)))))))))
if(x.lt.0)erfcc=2.0-erfcc

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

subroutine leastsq(x,y,n,miss,a,b,r)
! --------------------------------------------------------------------
! subroutine to calculate linear regression
! calculates the regression constant, regression
! coefficient and r**2
! --------------------------------------------------------------------

! ***input***
! x(n) [real] = x time series
! y(n) [real] = y time series
! n [integer] = length of x
! miss [real] = missing value
! ***output***
! a [real] = regression constant
! b [real] = regression coefficient
! r [real] = r-squared error

implicit none

integer n,i,nm
real x(n),y(n),a,b,r,miss
real sx,sy,sxy,sx2,sy2

! calculate regression constant (a) and coefficient (b)
sxy=0.0
sx=0.0
sy=0.0
sx2=0.0
sy2=0.0
nm=0
do i=1,n
 if(x(i).ne.miss.and.y(i).ne.miss)then
  sxy=sxy+x(i)*y(i)
  sx=sx+x(i)
  sy=sy+y(i)
  sx2=sx2+x(i)*x(i)
  sy2=sy2+y(i)*y(i)
  nm=nm+1
 end if
end do
if(nm.lt.2)then
 a=miss
 b=miss
 r=miss
else
 b=(nm*sxy-sx*sy)/(nm*sx2-sx*sx)
 a=(sy-sx*b)/real(nm)
end if

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

function lngstr( string )
character string*(*)
ilen = len(string)
do 100 lngstr=ilen,1,-1
 if ( string(lngstr:lngstr) .ne. ' ' ) go to 99
100 continue
lngstr = 0
99 continue
return
end
