subroutine extremes_indices()

! version 3.3.1
! see accompanying documentation

! calculate extreme indices from daily data for a single location
! data are loaded using a common block (not ideal but that's the way the earliest version was built)
! see indices.inc for a description of what variables to fill

include 'indices.inc'

real pww_arr(minyr:maxyr,5),pdd_arr(minyr:maxyr,5)

! do some checks
if(minyr.gt.maxyr)stop 'indices error: minyr > maxyr'
if(bm_minyr.gt.bm_maxyr)stop 'indices error: bm_minyr > bm_maxyr'
if(minyr.gt.bm_minyr)stop 'indices error: bm_minyr < minyr'
if(maxyr.lt.bm_maxyr)stop 'indices error: bm_maxyr < maxyr'

! inits
call init_ts()

! calculate fraction of non-missing data
call fraction

! normals
call normals()

! Indices

! mean temperatures
indexID=indexID+1
call calc_mean(2)
names(indexID)='txav'
indexID=indexID+1
call calc_mean(3)
names(indexID)='tnav'
indexID=indexID+1
call calc_mean(4)
names(indexID)='tav'

! Diurnal temperature range
call temp_range()

indexID=indexID+1
call t_perc(10.0,2) !Tmax 10th percentile
names(indexID)='tmax10p'
if(newnames)names(indexID)= 'txq10'

indexID=indexID+1
call t_perc(90.0,2) !Tmax 90th percentile
names(indexID)='tmax90p'
if(newnames)names(indexID)= 'txq90'

indexID=indexID+1
call t_perc(10.0,3) !Tmin 10th percentile
names(indexID)='tmin10p'
if(newnames)names(indexID)= 'tnq10'

indexID=indexID+1
call t_perc(90.0,3) !Tmin 90th percentile
names(indexID)='tmin90p'
if(newnames)names(indexID)= 'tnq90'

indexID=indexID+1
call Below_Thresh(3,0.0) !number of frost days (TMIN <=0)
names(indexID)='125Fd'
if(newnames)names(indexID)= 'tnfd'

indexID=indexID+1
call Below_Thresh(2,0.0) !number of ice days (TMAX <=0)
names(indexID)='114Id'
if(newnames)names(indexID)= 'txice'

indexID=indexID+1
call DegDays() ! Growing degree days > DegDays_Thresh degC
names(indexID)= '135GD'
if(newnames)names(indexID)= 'tgdd'

indexID=indexID+1
call ETR ! intra-annual extreme temperature range
names(indexID)='141ETR'
if(newnames)names(indexID)= 'tiaetr'

indexID=indexID+1
call GSL() ! Growing Season Length
names(indexID)= '143GSL'
if(newnames)names(indexID)= 'tgsl'

indexID=indexID+1
call hcwdi(5.0,5,.true.,2)     ! heat wave duration index
names(indexID)= '144HWDI'
if(newnames)names(indexID)= 'txhwd'

indexID=indexID+1
call hcwdi_perc(90.0,.true.,2)     ! max heat wave percentile index
names(indexID)= 'txhw90'

indexID=indexID+1
call hcwdi(5.0,5,.false.,3)     ! cold wave duration index
names(indexID)= '145CWDI'
if(newnames)names(indexID)= 'tncwd'

indexID=indexID+1
call hcwdi_perc(10.0,.false.,3)     ! max cold wave percentile index
names(indexID)= 'tncw10'

indexID=indexID+1
call FSL(0.0) ! Frost Season Length 0 deg
names(indexID)= '147FSL0'
if(newnames)names(indexID)= 'tnfsl'

indexID=indexID+1
call Tx10  !percent of time Tmax below 10th percentile
names(indexID)='191Tx10'
if(newnames)names(indexID)= 'txf10'

indexID=indexID+1
call Tx90  !percent of time Tmax above 90th percentile
names(indexID)='192Tx90'
if(newnames)names(indexID)= 'txf90'

indexID=indexID+1
call Tn10  !percent of time Tmin below 10th percentile
names(indexID)='193Tn10'
if(newnames)names(indexID)= 'tnf10'

indexID=indexID+1
call Tn90  !percent of time Tmin above 90th percentile
names(indexID)='194Tn90'
if(newnames)names(indexID)= 'tnf90'

!Mean climatological precipitation (mm/day)
indexID=indexID+1
call calc_mean(1)
names(indexID)= '601R'
if(newnames)names(indexID)= 'pav'

!Quantiles (20,40,50,60,80,95%) of rainday ammounts (mm/day)
indexID=indexID+1
call prec_perc(20.0)
names(indexID)= 'prec20p'
if(newnames)names(indexID)= 'pq20'

indexID=indexID+1
call prec_perc(40.0)
names(indexID)= 'prec40p'
if(newnames)names(indexID)= 'pq40'

indexID=indexID+1
call prec_perc(50.0)
names(indexID)= 'prec50p'
if(newnames)names(indexID)= 'pq50'

indexID=indexID+1
call prec_perc(60.0)
names(indexID)= 'prec60p'
if(newnames)names(indexID)= 'pq60'

indexID=indexID+1
call prec_perc(80.0)
names(indexID)= 'prec80p'
if(newnames)names(indexID)= 'pq80'

indexID=indexID+1
call prec_perc(90.0)
names(indexID)= 'prec90p'
if(newnames)names(indexID)= 'pq90'

indexID=indexID+1
call prec_perc(95.0)
names(indexID)= 'prec95p'
if(newnames)names(indexID)= 'pq95'

!Fraction of total precipitation above annual quantiles
indexID=indexID+1
call prec_quant(20.0)
names(indexID)= 'frac20p'
if(newnames)names(indexID)= 'pf20'

indexID=indexID+1
call prec_quant(40.0)
names(indexID)= 'frac40p'
if(newnames)names(indexID)= 'pf40'

indexID=indexID+1
call prec_quant(50.0)
names(indexID)= 'frac50p'
if(newnames)names(indexID)= 'pf50'

indexID=indexID+1
call prec_quant(60.0)
names(indexID)= 'frac60p'
if(newnames)names(indexID)= 'pf60'

indexID=indexID+1
call prec_quant(80.0)
names(indexID)= 'frac80p'
if(newnames)names(indexID)= 'pf80'

indexID=indexID+1
call prec_quant(90.0)
names(indexID)= 'frac90p'
if(newnames)names(indexID)= 'pf90'

indexID=indexID+1
call prec_quant(95.0)
names(indexID)= 'frac95p'
if(newnames)names(indexID)= 'pf95'

indexID=indexID+1
call Above_Thresh(1,10.0)  ! number of days >= 10mm
names(indexID)='606R10'
if(newnames)names(indexID)= 'pn10mm'

indexID=indexID+1
call cdd  !consecutive dry days subroutine
names(indexID)= '641CDD'
if(newnames)names(indexID)= 'pxcdd'

indexID=indexID+1
call cwd  !consecutive wet days subroutine
names(indexID)= '642CWD'
if(newnames)names(indexID)= 'pxcwd'

!Mean wet-day persistence (Pww)
indexID=indexID+1
call prec_pww(pww_arr)
names(indexID)='pww'
if(newnames)names(indexID)= 'ppww'

!Mean dry-day persistence (Pdd)
indexID=indexID+1
call prec_pdd(pdd_arr)
names(indexID)='persist_dd'
if(newnames)names(indexID)= 'ppdd'

indexID=indexID+1
call persist_corr(pww_arr,pdd_arr)
names(indexID)='persist_corr'
if(newnames)names(indexID)= 'ppcr'

! stats of dry spell lengths (days)
call wet_spell_stats()

! stats of dry spell lengths (days)
call dry_spell_stats()

indexID=indexID+1
call r3_5_10d(3) ! greatest 3 day rainfall total
names(indexID)='643R3d'
if(newnames)names(indexID)= 'px3d'

indexID=indexID+1
call r3_5_10d(5) ! greatest 5 day rainfall total
names(indexID)='644R5d'
if(newnames)names(indexID)= 'px5d'

indexID=indexID+1
call r3_5_10d(10) ! greatest 10 day rainfall total
names(indexID)='645R10d'
if(newnames)names(indexID)= 'px10d'

indexID=indexID+1
call sdii !simple daily precip intensity index
names(indexID)='646SDII'
if(newnames)names(indexID)= 'pint'

indexID=indexID+1
call RThreshT(90.0) ! Fraction of Ann. Total Precip due to events above long-term 90th %tile
names(indexID)='691R90T'
if(newnames)names(indexID)= 'pfl90'

indexID=indexID+1
call RThreshN(90.0) ! No. of events above long-term 90th %tile
names(indexID)='692R90N'
if(newnames)names(indexID)= 'pnl90'

call cleanup()

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! START OF SUBROUTINES
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

subroutine init_ts()
include 'indices.inc'
integer i,j,k,realdaysinmth
integer yr,mth,day,ivar,issn,iyr,iday,ssn(12)

miss_internal=-9999.9
indexID=0
ssn(1)=1
ssn(2)=1
ssn(3)=2
ssn(4)=2
ssn(5)=2
ssn(6)=3
ssn(7)=3
ssn(8)=3
ssn(9)=4
ssn(10)=4
ssn(11)=4
ssn(12)=1

! set all missing values to internal missing value
do yr=minyr,maxyr
 do mth=1,12
  do day=1,31
   do ivar=1,4
    if(data_in(yr,mth,day,ivar).eq.miss)data_in(yr,mth,day,ivar)=miss_internal
   end do
  end do
 end do
end do

! init seasonal array
do ivar=1,4
 do yr=minyr,maxyr
  do issn=1,4
   do day=1,93
    data_ssn(yr,issn,day,ivar)=miss_internal
   end do
  end do
 end do
end do

! load seasonal array
do ivar=1,4
! first December is missing
 issn=1
 iday=0
 do day=1,31
  iday=iday+1
  data_ssn(minyr,issn,iday,ivar)=miss_internal
 end do
 do yr=minyr,maxyr
  do mth=1,12
   if(.not.(yr.eq.maxyr.and.mth.eq.12))then
    if(ssn(mth).ne.issn)iday=0
    issn=ssn(mth)
    if(mth.eq.12)then
     iyr=yr+1
    else
     iyr=yr
    end if
    do day=1,realdaysinmth(yr,mth)
     iday=iday+1
     data_ssn(iyr,issn,iday,ivar)=data_in(yr,mth,day,ivar)
    end do
   end if
  end do
 end do
end do

! init results
do k=1,nIndices
 do i=minyr,maxyr
  do j=1,5
   ts(k,i,j)=miss_internal
  enddo
 enddo
enddo

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

subroutine cleanup()
include 'indices.inc'
integer yr,mth,day,ivar,id

! reset missing values
do yr=minyr,maxyr
 do mth=1,12
  do day=1,31
   do ivar=1,4
    if(data_in(yr,mth,day,ivar).eq.miss_internal)data_in(yr,mth,day,ivar)=miss
   end do
  end do
 end do
end do
do id=1,nIndices
 do yr=minyr,maxyr
  do mth=1,5
   if(ts(id,yr,mth).eq.miss_internal)ts(id,yr,mth)=miss
  end do
 end do
end do

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!! start of temperature stats subs !!!!!!!!!!!!!!!

subroutine calc_mean(ivar)

include 'indices.inc'
integer i,j,k,n,ivar
real arr(366)
integer realdaysinmth,realdaysinssn

! seasonal 
do i=minyr,maxyr
 do j=1,4
  if (frac_ssn(i,j,ivar) .ge. frac_thresh) then
   n=realdaysinssn(i,j)
   do k=1,n
    arr(k)=data_ssn(i,j,k,ivar)
   enddo
   call average(n,miss_internal,arr,ts(indexID,i,j))
  endif
 enddo
enddo

! annual
do i=minyr,maxyr
 if (frac(i,13,ivar) .ge. frac_thresh) then
  n=0
  do j=1,12
   do k=1,realdaysinmth(i,j)
    n=n+1
    arr(n)=data_in(i,j,k,ivar)
   enddo
  enddo
  call average(n,miss_internal,arr,ts(indexID,i,5))
 endif
enddo

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

! Diurnal temperature range
subroutine temp_range()

include 'indices.inc'

real p,quickpile,mean,total
integer i,j,k,m,n
integer realdaysinmth
real arr(400)
real ts_mean(minyr:maxyr,5)
real ts_10perc(minyr:maxyr,5)
real ts_90perc(minyr:maxyr,5)

logical frac_data

do i=minyr,maxyr
 n = 0
 if(i.eq.minyr)then
  frac_data = .false.
 elseif (((frac(i-1,12,2)+frac(i,1,2)+frac(i,2,2))/3.0) .ge. frac_thresh &
 .and. ((frac(i-1,12,3)+frac(i,1,3)+frac(i,2,3))/3.0) .ge. frac_thresh) then
  frac_data = .true.
 else
  frac_data = .false.
 endif

! calc for DJF
 if (frac_data) then
  do k=1,realdaysinmth(i-1,12)
   if (data_in(i-1,12,k,2) .ne.miss_internal .and. data_in(i-1,12,k,3) .ne.miss_internal) then !data present
    n=n+1
    arr(n)=data_in(i-1,12,k,2) - data_in(i-1,12,k,3)
   endif
  enddo

  do j=1,2
   do k=1,realdaysinmth(i,j)
! test for a consecutive rain day
    if (data_in(i,j,k,2) .ne.miss_internal .and. data_in(i,j,k,3) .ne.miss_internal) then
     n=n+1
     arr(n) = data_in(i,j,k,2) - data_in(i,j,k,3)
    endif
   enddo
  enddo

! calculate the mean for DJF
  total = 0
  do m=1,n
   total = total + arr(m)
  enddo
  mean = total / real(n)
  ts_mean(i,1)=mean

! calculate the 10th percentile for DJF
  p=quickpile(arr,n,10.0,miss_internal)
  ts_10perc(i,1)=p

! calculate the 90th percentile for DJF
  p=quickpile(arr,n,90.0,miss_internal)
  ts_90perc(i,1)=p

 else
  ts_mean(i,1)=miss_internal
  ts_10perc(i,1)=miss_internal
  ts_90perc(i,1)=miss_internal
 endif

 n=0

! calc for MAM
 if (((frac(i,3,2)+frac(i,4,2)+frac(i,5,2))/3.0) .ge. frac_thresh &
  .and. ((frac(i,3,3)+frac(i,4,3)+frac(i,5,3))/3.0) .ge. frac_thresh) then
  frac_data = .true.
 else
  frac_data = .false.
 endif

 if (frac_data) then
  do j=3,5
   do k=1,realdaysinmth(i,j)
    if (data_in(i,j,k,2) .ne.miss_internal .and. data_in(i,j,k,3) .ne.miss_internal) then !data present
     n=n+1
     arr(n) = data_in(i,j,k,2) - data_in(i,j,k,3)
    endif
   enddo
  enddo

! calculate the mean for MAM
  total = 0
  do m=1,n
   total = total + arr(m)
  enddo
  mean = total / real(n)
  ts_mean(i,2)=mean

! calculate the 10th percentile for MAM
  p=quickpile(arr,n,10.0,miss_internal)
  ts_10perc(i,2)=p

! calculate the 90th percentile for MAM
  p=quickpile(arr,n,90.0,miss_internal)
  ts_90perc(i,2)=p

 else
  ts_mean(i,2)=miss_internal
  ts_10perc(i,2)=miss_internal
  ts_90perc(i,2)=miss_internal
 endif

 n=0

! calc for JJA
 if (((frac(i,6,2)+frac(i,7,2)+frac(i,8,2))/3.0) .ge. frac_thresh &
  .and. ((frac(i,6,3)+frac(i,7,3)+frac(i,8,3))/3.0) .ge. frac_thresh) then
  frac_data = .true.
 else
  frac_data = .false.
 endif
 if (frac_data) then
  do j=6,8
   do k=1,realdaysinmth(i,j)
    if (data_in(i,j,k,2) .ne.miss_internal .and. data_in(i,j,k,3) .ne.miss_internal) then ! data present
     n=n+1
     arr(n) = data_in(i,j,k,2) - data_in(i,j,k,3)
    endif
   enddo
  enddo

! calculate the mean for JJA
  total = 0
  do m=1,n
   total = total + arr(m)
  enddo
  mean = total / real(n)
  ts_mean(i,3)=mean

! calculate the 10th percentile for JJA
  p=quickpile(arr,n,10.0,miss_internal)
  ts_10perc(i,3)=p

! calculate the 90th percentile for JJA
  p=quickpile(arr,n,90.0,miss_internal)
  ts_90perc(i,3)=p

 else
  ts_mean(i,3)=miss_internal
  ts_10perc(i,3)=miss_internal
  ts_90perc(i,3)=miss_internal
 endif

 n=0

! calc for SON
 if (((frac(i,9,2)+frac(i,10,2)+frac(i,11,2))/3.0) .ge. frac_thresh &
 .and. ((frac(i,9,3)+frac(i,10,3)+frac(i,11,3))/3.0) .ge. frac_thresh) then
  frac_data = .true.
 else
  frac_data = .false.
 endif
 if (frac_data) then
  do j=9,11
   do k=1,realdaysinmth(i,j)
    if (data_in(i,j,k,2) .ne.miss_internal .and. data_in(i,j,k,3) .ne.miss_internal) then ! data present
     n=n+1
     arr(n) = data_in(i,j,k,2) - data_in(i,j,k,3)
    endif
   enddo
  enddo

! calculate the mean for SON
  total = 0
  do m=1,n
   total = total + arr(m)
  enddo
  mean = total / real(n)
  ts_mean(i,4)=mean

! calculate the 10th percentile for SON
  p=quickpile(arr,n,10.0,miss_internal)
  ts_10perc(i,4)=p

! calculate the 90th percentile for SON
  p=quickpile(arr,n,90.0,miss_internal)
  ts_90perc(i,4)=p

 else
  ts_mean(i,4)=miss_internal
  ts_10perc(i,4)=miss_internal
  ts_90perc(i,4)=miss_internal
 endif

 n=0

enddo

! calc the annual
do i=minyr,maxyr
 n=0
 if (frac(i,13,2) .ge. frac_thresh .and. frac(i,13,3) .ge. frac_thresh) then
  frac_data = .true.
 else
  frac_data = .false.
 endif
 if (frac_data) then
  do j=1,12
   do k=1,realdaysinmth(i,j)
    if (data_in(i,j,k,2) .ne.miss_internal .and. data_in(i,j,k,3) .ne.miss_internal) then ! data present
     n=n+1
     arr(n) = data_in(i,j,k,2) - data_in(i,j,k,3)
    endif
   enddo
  enddo

! calculate the mean for ANNUAL
  total = 0
  do m=1,n
   total = total + arr(m)
  enddo
  mean = total / real(n)
  ts_mean(i,5)=mean

! calculate the 10th percentile for ANNUAL
  p=quickpile(arr,n,10.0,miss_internal)
  ts_10perc(i,5)=p

! calculate the 90th percentile for ANNUAL
  p=quickpile(arr,n,90.0,miss_internal)
  ts_90perc(i,5)=p

 else
  ts_mean(i,5)=miss_internal
  ts_10perc(i,5)=miss_internal
  ts_90perc(i,5)=miss_internal
 endif
enddo

! write the results for the mean
indexID=indexID+1
names(indexID)= 'Trange_mean'
if(newnames)names(indexID)= 'trav'
do i=minyr,maxyr
 do j=1,5
  ts(indexID,i,j)=ts_mean(i,j)
 enddo
enddo

! write the results for the 10th percentile
indexID=indexID+1
names(indexID)= 'Trange10p'
if(newnames)names(indexID)= 'trq10'
do i=minyr,maxyr
 do j=1,5
  ts(indexID,i,j)=ts_10perc(i,j)
 enddo
enddo

! write the results for the 90th percentile
indexID=indexID+1
names(indexID)= 'Trange90p'
if(newnames)names(indexID)= 'trq90'
do i=minyr,maxyr
 do j=1,5
  ts(indexID,i,j)=ts_90perc(i,j)
 enddo
enddo

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

subroutine t_perc(percent,ivar)

include 'indices.inc'
integer i,j,k,n,ivar
real arr(366)
real quickpile, percent
integer realdaysinmth,realdaysinssn

! seasonal 
do i=minyr,maxyr
 do j=1,4
  if (frac_ssn(i,j,ivar) .ge. frac_thresh) then
   n=0
   do k=1,realdaysinssn(i,j)
    if(data_ssn(i,j,k,ivar) .ne.miss_internal)then !data present
     n=n+1
     arr(n)=data_ssn(i,j,k,ivar)
    endif
   enddo
   ts(indexID,i,j)=quickpile(arr,n,percent,miss_internal)
  endif
 enddo
enddo

! annual
do i=minyr,maxyr
 if (frac(i,13,ivar) .ge. frac_thresh) then
  n=0
  do j=1,12
   do k=1,realdaysinmth(i,j)
    if(data_in(i,j,k,ivar) .ne.miss_internal)then !data present
     n=n+1
     arr(n)=data_in(i,j,k,ivar)
    endif
   enddo
  enddo
  ts(indexID,i,5)=quickpile(arr,n,percent,miss_internal)
 endif
enddo

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

  subroutine Below_Thresh(ivar,thresh)
! this index calculates the number of days with var <= thresh

include 'indices.inc'
integer i,j,k,ivar
integer muse(12)
real dnum,thresh
integer realdaysinmth,realdaysinssn

! seasonal
do i=minyr,maxyr
 do j=1,4
  if(frac_ssn(i,j,ivar).ge.frac_thresh)then
   dnum=0
   do k=1,realdaysinssn(i,j)
    if(data_ssn(i,j,k,ivar).ne.miss_internal)then !data present
     if(data_ssn(i,j,k,ivar).le.thresh)then
      dnum=dnum+1
     endif
    endif
   enddo
   ts(indexID,i,j)=dnum
  end if
 enddo
enddo

! annual
do i=1,12
 muse(i)=0
enddo
do i=minyr,maxyr
 if(frac(i,13,ivar).ge.frac_thresh)then
  dnum=0
  do j=1,12
   do k=1,realdaysinmth(i,j)
    if(data_in(i,j,k,ivar).ne.miss_internal)then !data present
     if(data_in(i,j,k,ivar).le.thresh)then
      dnum=dnum+1
      muse(j)=1
     endif
    endif
   enddo
  enddo
  ts(indexID,i,5)=dnum
 end if
enddo

call normalize(indexID,muse,ivar)

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

subroutine DegDays() ! Index 134 Thermal degree days

include 'indices.inc'
integer i,endyear
integer ibegdy,ienddy
real dd

! SH stations are done July-June
if(south)then
 endyear=maxyr-1
else
 endyear=maxyr
endif

do i=minyr,endyear
 call GetGrowing(i,ibegdy,ienddy,dd)
 if (ibegdy.eq.0.or.ienddy.eq.0) then
  ts(indexID,i,5) = miss_internal
 else
  ts(indexID,i,5)=dd
 endif
enddo

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

subroutine etr
! this index calculates the intra-annual (or seasonal) extreme temperature range

include 'indices.inc'
integer i,j,k
integer muse1(12),muse2(12)
integer mmin,mmax,mzip
real tmin,tmax
integer realdaysinmth,realdaysinssn

! seasonal

do i=minyr,maxyr
 do j=1,4
  if(frac_ssn(i,j,2).ge.frac_thresh.and.frac_ssn(i,j,3).ge.frac_thresh)then
   tmin=99999
   tmax=-99999
   do k=1,realdaysinssn(i,j)
    if(data_ssn(i,j,k,2).ne.miss_internal)tmax=max(tmax,data_ssn(i,j,k,2))
    if(data_ssn(i,j,k,3).ne.miss_internal)tmin=min(tmin,data_ssn(i,j,k,3))
   enddo
   ts(indexID,i,j)=tmax-tmin
  endif
 enddo
enddo

! annual

do j=1,12
 muse1(j)=0
 muse2(j)=0
enddo

do i=minyr,maxyr
 tmin=99999
 tmax=-99999
 mmin=0
 mmax=0
 do j=1,12
  do k=1,realdaysinmth(i,j)
   if(data_in(i,j,k,2) > tmax)mmax=j
   if(data_in(i,j,k,2) .ne.miss_internal)tmax=max(tmax,data_in(i,j,k,2))
   if(data_in(i,j,k,3).lt.tmin.and.data_in(i,j,k,3) .ne.miss_internal)mmin=j
   if(data_in(i,j,k,3) .ne.miss_internal)tmin=min(tmin,data_in(i,j,k,3))
  enddo
 enddo
 if (mmax.ne.0.and.mmin.ne.0) then
   muse1(mmax)=1
   muse2(mmin)=1
   ts(indexID,i,5)=tmax-tmin
 endif
enddo

! remove years with too little data during cold/warm months
do i=minyr,maxyr
 mzip=0
 do j=1,12
  if(muse1(j).eq.1)then
   if(frac(i,j,2).lt.frac_thresh)mzip=1
  endif
  if(muse2(j).eq.1)then
   if(frac(i,j,3).lt.frac_thresh)mzip=1
  endif
 enddo
 if(mzip.eq.1)ts(indexID,i,5)=miss_internal
enddo

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

Subroutine GSL() ! Index 143 Thermal Growing Season

include 'indices.inc'
integer i,endyear
integer ibegdy,ienddy
real dd

! SH stations are done July-June
if(south)then
 endyear=maxyr-1
else
 endyear=maxyr
endif

do i=minyr,endyear
 call GetGrowing(i,ibegdy,ienddy,dd)
 if (ibegdy.eq.0.or.ienddy.eq.0) then
  ts(indexID,i,5) = miss_internal
 else
   ts(indexID,i,5) = ienddy-ibegdy
 endif

enddo

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

subroutine hcwdi(AnomalyThresh,DaysThresh,hot,ivar)
! this index calculates the heat or cold wave duration index

include 'indices.inc'
integer i,j,k
integer ConsecDayCounter,HeatWaveDays
real Threshold,AnomalyThresh
integer DaysThresh,FirstDayOfHeatWave
integer realdaysinmth,realdaysinssn,ivar
Logical hot ! .true. for heat wave, otherwise cold

! Parameters

! Now loop through all years for a station and compare daily value to
! the 'DailyAverage' array to find anomalies.  If more than 'DaysThresh'
! days in a row where TMAX or TMIN anomalies are > or < 'AnomalyThresh',
! then a heat or cold wave exists.

! seasonal 
do i=minyr,maxyr
 do j=1,4
  ConsecDayCounter = 0
  HeatWaveDays = 0
  do k=1,realdaysinssn(i,j)
   if ((DailyNormals_ssn(j,k,ivar) .ne.miss_internal).and.(data_ssn(i,j,k,ivar) .ne.miss_internal)) then
    if(hot)then
     Threshold = DailyNormals_ssn(j,k,ivar) + AnomalyThresh
     if (data_ssn(i,j,k,ivar) > Threshold) then
      ConsecDayCounter=ConsecDayCounter+1
     else
      ConsecDayCounter=0
     endif
    else
     Threshold = DailyNormals_ssn(j,k,ivar) - AnomalyThresh
     if (data_ssn(i,j,k,ivar) < Threshold) then
      ConsecDayCounter=ConsecDayCounter+1
     else
      ConsecDayCounter=0
     endif
    end if
    if (ConsecDayCounter > DaysThresh) then
     FirstDayOfHeatWave = DaysThresh+1
     if (ConsecDayCounter.eq.FirstDayOfHeatWave) then
      HeatWaveDays = HeatWaveDays+DaysThresh+1
     else
      HeatWaveDays = HeatWaveDays+1
     endif
    endif
   endif
  enddo
  if (frac_ssn(i,j,ivar).ge.frac_thresh) then
   ts(indexID,i,j) = HeatWaveDays
  else
   ts(indexID,i,j) = miss_internal
  endif
 enddo
enddo

! annual
do i=minyr,maxyr
 ConsecDayCounter = 0
 HeatWaveDays = 0
 do j=1,12
  do k=1,realdaysinmth(i,j)
   if ((DailyNormals(j,k,ivar) .ne.miss_internal).and.(data_in(i,j,k,ivar) .ne.miss_internal)) then
    if(hot)then
     Threshold = DailyNormals(j,k,ivar) + AnomalyThresh
     if (data_in(i,j,k,ivar) > Threshold) then
      ConsecDayCounter=ConsecDayCounter+1
     else
      ConsecDayCounter=0
     endif
    else
     Threshold = DailyNormals(j,k,ivar) - AnomalyThresh
     if (data_in(i,j,k,ivar) < Threshold) then
      ConsecDayCounter=ConsecDayCounter+1
     else
      ConsecDayCounter=0
     endif
    end if
    if (ConsecDayCounter > DaysThresh) then
     FirstDayOfHeatWave = DaysThresh+1
     if (ConsecDayCounter.eq.FirstDayOfHeatWave) then
      HeatWaveDays = HeatWaveDays+DaysThresh+1
     else
      HeatWaveDays = HeatWaveDays+1
     endif
    endif
   endif
  enddo
 enddo
 if (frac(i,13,ivar).ge.frac_thresh) then
  ts(indexID,i,5) = HeatWaveDays
 else
  ts(indexID,i,5) = miss_internal
 endif
enddo

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

subroutine hcwdi_perc(AnomalyThresh,hot,ivar)
! this index calculates the heat or cold wave duration index besed on percentile thresholds
! it looks for the max no. of consecutive days above or below the threshold

include 'indices.inc'
integer i,j,k
integer ConsecDayCounter,HeatWaveDays
real Threshold,AnomalyThresh
integer realdaysinmth,realdaysinssn,ivar
Logical hot ! .true. for heat wave, otherwise cold

! calculate percentile-based normals
call normals_perc(AnomalyThresh)

! seasonal 
do i=minyr,maxyr
 do j=1,4
  ConsecDayCounter = 0
  HeatWaveDays = 0
  do k=1,realdaysinssn(i,j)
   if ((DailyNormals_ssn_perc(j,k,ivar) .ne.miss_internal).and.(data_ssn(i,j,k,ivar) .ne.miss_internal)) then
    Threshold = DailyNormals_ssn_perc(j,k,ivar)
    if(hot)then
     if (data_ssn(i,j,k,ivar) > Threshold) then
      ConsecDayCounter=ConsecDayCounter+1
     else
      ConsecDayCounter=0
     endif
    else
     if (data_ssn(i,j,k,ivar) < Threshold) then
      ConsecDayCounter=ConsecDayCounter+1
     else
      ConsecDayCounter=0
     endif
    end if
    if (ConsecDayCounter > HeatWaveDays)HeatWaveDays=ConsecDayCounter
   else
    ConsecDayCounter=0 ! missing day breaks heat wave
   endif
  enddo
  if (frac_ssn(i,j,ivar).ge.frac_thresh) then
   ts(indexID,i,j) = HeatWaveDays
  else
   ts(indexID,i,j) = miss_internal
  endif
 enddo
enddo

! annual
do i=minyr,maxyr
 ConsecDayCounter = 0
 HeatWaveDays = 0
 do j=1,12
  do k=1,realdaysinmth(i,j)
   if ((DailyNormals_perc(j,k,ivar) .ne.miss_internal).and.(data_in(i,j,k,ivar) .ne.miss_internal)) then
    Threshold = DailyNormals_perc(j,k,ivar)
    if(hot)then
     if (data_in(i,j,k,ivar) > Threshold) then
      ConsecDayCounter=ConsecDayCounter+1
     else
      ConsecDayCounter=0
     endif
    else
     if (data_in(i,j,k,ivar) < Threshold) then
      ConsecDayCounter=ConsecDayCounter+1
     else
      ConsecDayCounter=0
     endif
    end if
    if (ConsecDayCounter > HeatWaveDays)HeatWaveDays=ConsecDayCounter
   else
    ConsecDayCounter=0 ! missing day breaks heat wave
   endif
  enddo
 enddo
 if (frac(i,13,ivar).ge.frac_thresh) then
  ts(indexID,i,5) = HeatWaveDays
 else
  ts(indexID,i,5) = miss_internal
 endif
enddo

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

Subroutine FSL(thresh) ! Frost season length

include 'indices.inc'
integer i,j,k,endyear,numdays,yr,mth
integer iflagb,ibegdy,iflage,ienddy
real tmin(366),thresh
integer realdaysinmth,nonmiss

! NH stations are done July-June
if(south)then
 endyear=maxyr
else
 endyear=maxyr-1
endif

do i=minyr,endyear
 numdays=0
 nonmiss=0
 yr=i
 if(south)then
  mth=0
 else
  mth=6
 endif
 do j=1,12
  mth=mth+1
  if(mth > 12)then
   mth=1
   yr=i+1
  endif
  do k=1,realdaysinmth(yr,mth)
   numdays=numdays+1
   tmin(numdays)=data_in(yr,mth,k,3)
   if(tmin(numdays).ne.miss_internal)nonmiss=nonmiss+1
  enddo
 enddo

! Determine the start of the Frost Season

 iflagb = 0
 ibegdy = 0
 do k=1,numdays
  if(tmin(k).lt.thresh.and.tmin(k) .ne.miss_internal.and.iflagb.eq.0) then
   ibegdy = k
   iflagb = 1
  endif
 enddo

! Determine the end of the Frost Season

 iflage = 0
 ienddy = 0
 do k=numdays,1,-1
  if(tmin(k).lt.thresh.and.tmin(k) .ne.miss_internal.and.iflage.eq.0) then
   ienddy = k
   iflage = 1
  endif
 enddo

 if (iflagb.eq.0.or.iflage.eq.0.or.nonmiss<365*frac_thresh) then
  ts(indexID,i,5) = miss_internal
 else
   ts(indexID,i,5) = ienddy-ibegdy
 endif

enddo

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

subroutine Tx10
!the % of time Tmax < 10th percentile (percentiles based on 61-90)

include 'indices.inc'
integer i,j,k,n
real arr(nyrs*100),perc_arr(minyr:maxyr,12)
real p10,z,z1,z2,zc
real quickpile
integer realdaysinmth

do j=1,12
 n=0
 do i=bm_minyr,bm_maxyr
  do k=1,realdaysinmth(i,j)
   if(data_in(i,j,k,2) .ne.miss_internal)then !data present
    n=n+1
    arr(n)=data_in(i,j,k,2)
   endif
  enddo
 enddo
 p10=quickpile(arr,n,10.0,miss_internal)
! p10 is now the value of the 10th percentile
 do i=minyr,maxyr
  z1=0
  z2=0
  do k=1,realdaysinmth(i,j)
   if(data_in(i,j,k,2) .ne.miss_internal)then !data present
    if(data_in(i,j,k,2).lt.p10)then
     z1=z1+1
    else
     z2=z2+1
    endif
   endif
  enddo
  z=miss_internal
  if(z1+z2 > 0)z=z1/(z1+z2)
  if(frac(i,j,2).lt.frac_thresh)z=miss_internal
  perc_arr(i,j)=z
 enddo
enddo

! calculate for DJF
do i=minyr,maxyr
 z=0 !value
 zc=0 !counter

 if (i /= minyr) then
  if (perc_arr(i-1,12) .ne.miss_internal) then
   z=z+perc_arr(i-1,12)
   zc=zc+1
  endif

  do j=1,2
   if (perc_arr(i,j) .ne.miss_internal) then
    z=z+perc_arr(i,j)
    zc=zc+1
   endif
  enddo

  if (((frac(i-1,12,2)+frac(i,1,2)+frac(i,2,2))/3.0).lt.frac_thresh) then
   ts(indexID,i,1)=miss_internal
  else
   ts(indexID,i,1)=z/zc
  endif
 endif
enddo

! calculate for MAM
do i=minyr,maxyr
 z=0 !value
 zc=0 !counter
 do j=3,5
  if (perc_arr(i,j) .ne.miss_internal) then
   z=z+perc_arr(i,j)
   zc=zc+1
  end if
 end do
 if (((frac(i,3,2)+frac(i,4,2)+frac(i,5,2))/3.0).lt.frac_thresh) then
  ts(indexID,i,2)=miss_internal
 else
  ts(indexID,i,2)=z/zc
 end if
end do

! calculate for JJA
do i=minyr,maxyr
 z=0 !value
 zc=0 !counter
 do j=6,8
  if (perc_arr(i,j) .ne.miss_internal) then
   z=z+perc_arr(i,j)
   zc=zc+1
  end if
 end do
 if (((frac(i,6,2)+frac(i,7,2)+frac(i,8,2))/3.0).lt.frac_thresh) then
  ts(indexID,i,3)=miss_internal
 else
  ts(indexID,i,3)=z/zc
 end if
end do

! calculate for SON
do i=minyr,maxyr
 z=0 !value
 zc=0 !counter
 do j=9,11
  if (perc_arr(i,j) .ne.miss_internal) then
   z=z+perc_arr(i,j)
   zc=zc+1
  endif
 enddo
 if (((frac(i,9,2)+frac(i,10,2)+frac(i,11,2))/3.0).lt.frac_thresh) then
  ts(indexID,i,4)=miss_internal
 else
  ts(indexID,i,4)=z/zc
 endif
enddo

! calculate for ANNUAL
do i=minyr,maxyr
 z=0 !value
 zc=0 !counter
 do j=1,12
  if (perc_arr(i,j) .ne.miss_internal) then
   z=z+perc_arr(i,j)
   zc=zc+1
  endif
 enddo
 if(frac(i,13,2) < frac_thresh)then
  ts(indexID,i,5)=miss_internal
 else
  ts(indexID,i,5)=z/zc
 endif
enddo

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

  subroutine Tx90
!the % of time Tmax > 90th percentile (percentiles based on 61-90)

include 'indices.inc'
integer i,j,k,n
real arr(nyrs*100),perc_arr(minyr:maxyr,12)
real p90,z,z1,z2,zc
real quickpile
integer realdaysinmth

do j=1,12
 n=0
 do i=bm_minyr,bm_maxyr
  do k=1,realdaysinmth(i,j)
   if(data_in(i,j,k,2) .ne.miss_internal)then !data present
    n=n+1
    arr(n)=data_in(i,j,k,2)
   endif
  enddo
 enddo
 p90=quickpile(arr,n,90.0,miss_internal)
! p90 is now the value of the 90th percentile
 do i=minyr,maxyr
  z1=0
  z2=0
  do k=1,realdaysinmth(i,j)
   if(data_in(i,j,k,2) .ne.miss_internal)then !data present
    if(data_in(i,j,k,2) > p90)then
     z1=z1+1
    else
     z2=z2+1
    endif
   endif
  enddo
  z=miss_internal
  if(z1+z2 > 0)z=z1/(z1+z2)
  if(frac(i,j,2).lt.frac_thresh)z=miss_internal
  perc_arr(i,j)=z
 enddo
enddo

! calculate for DJF
do i=minyr,maxyr
 z=0 !value
 zc=0 !counter

 if (i /= minyr) then
  if (perc_arr(i-1,12) .ne.miss_internal) then
   z=z+perc_arr(i-1,12)
   zc=zc+1
  endif

  do j=1,2
   if (perc_arr(i,j) .ne.miss_internal) then
    z=z+perc_arr(i,j)
    zc=zc+1
   endif
  enddo

  if (((frac(i-1,12,2)+frac(i,1,2)+frac(i,2,2))/3.0).lt.frac_thresh) then
   ts(indexID,i,1)=miss_internal
  else
   ts(indexID,i,1)=z/zc
  endif
 endif
enddo

! calculate for MAM
do i=minyr,maxyr
 z=0 !value
 zc=0 !counter
 do j=3,5
  if (perc_arr(i,j) .ne.miss_internal) then
   z=z+perc_arr(i,j)
   zc=zc+1
  endif
 enddo
 if (((frac(i,3,2)+frac(i,4,2)+frac(i,5,2))/3.0).lt.frac_thresh) then
  ts(indexID,i,2)=miss_internal
 else
  ts(indexID,i,2)=z/zc
 endif
enddo

! calculate for JJA
do i=minyr,maxyr
 z=0 !value
 zc=0 !counter
 do j=6,8
  if(perc_arr(i,j) .ne.miss_internal)then
   z=z+perc_arr(i,j)
   zc=zc+1
  endif
 enddo
 if (((frac(i,6,2)+frac(i,7,2)+frac(i,8,2))/3.0).lt.frac_thresh) then
  ts(indexID,i,3)=miss_internal
 else
  ts(indexID,i,3)=z/zc
 endif
enddo

! calculate for SON
do i=minyr,maxyr
 z=0 !value
 zc=0 !counter
 do j=9,11
  if (perc_arr(i,j) .ne.miss_internal) then
   z=z+perc_arr(i,j)
   zc=zc+1
  endif
 enddo
 if (((frac(i,9,2)+frac(i,10,2)+frac(i,11,2))/3.0).lt.frac_thresh) then
  ts(indexID,i,4)=miss_internal
 else
  ts(indexID,i,4)=z/zc
 endif
enddo

! calculate for ANNUAL
do i=minyr,maxyr
 z=0 !value
 zc=0 !counter
 do j=1,12
  if (perc_arr(i,j) .ne.miss_internal) then
   z=z+perc_arr(i,j)
   zc=zc+1
  endif
 enddo
 if (frac(i,13,2) < frac_thresh) then
  ts(indexID,i,5)=miss_internal
 else
  ts(indexID,i,5)=z/zc
 endif
enddo
return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

  subroutine Tn10
!the % of time Tmin < 10th percentile (percentiles based on 61-90)

include 'indices.inc'
integer i,j,k,n
real arr(nyrs*100),perc_arr(minyr:maxyr,12)
real p10,z,z1,z2,zc
real quickpile
integer realdaysinmth

do j=1,12
 n=0
 do i=bm_minyr,bm_maxyr
  do k=1,realdaysinmth(i,j)
   if(data_in(i,j,k,3) .ne.miss_internal)then !data present
    n=n+1
    arr(n)=data_in(i,j,k,3)
   endif
  enddo
 enddo
 p10=quickpile(arr,n,10.0,miss_internal)
! p10 is now the value of the 10th percentile
 do i=minyr,maxyr
  z1=0
  z2=0
  do k=1,realdaysinmth(i,j)
   if(data_in(i,j,k,3) .ne.miss_internal)then !data present
    if(data_in(i,j,k,3).lt.p10)then
     z1=z1+1
    else
     z2=z2+1
    endif
   endif
  enddo
  z=miss_internal
  if(z1+z2 > 0)z=z1/(z1+z2)
  if(frac(i,j,3).lt.frac_thresh)z=miss_internal
  perc_arr(i,j)=z
 enddo
enddo

! calculate for DJF
do i=minyr,maxyr
 z=0 !value
 zc=0 !counter

 if (i /= minyr) then
  if (perc_arr(i-1,12) .ne.miss_internal) then
   z=z+perc_arr(i-1,12)
   zc=zc+1
  endif

  do j=1,2
   if (perc_arr(i,j) .ne.miss_internal) then
    z=z+perc_arr(i,j)
    zc=zc+1
   endif
  enddo

  if (((frac(i-1,12,3)+frac(i,1,3)+frac(i,2,3))/3.0).lt.frac_thresh) then
   ts(indexID,i,1)=miss_internal
  else
   ts(indexID,i,1)=z/zc
  endif
 endif
enddo

! calculate for MAM
do i=minyr,maxyr
 z=0 !value
 zc=0 !counter
 do j=3,5
  if(perc_arr(i,j) .ne.miss_internal)then
   z=z+perc_arr(i,j)
   zc=zc+1
  endif
 enddo
 if (((frac(i,3,3)+frac(i,4,3)+frac(i,5,3))/3.0).lt.frac_thresh) then
  ts(indexID,i,2)=miss_internal
 else
  ts(indexID,i,2)=z/zc
 endif
enddo

! calculate for JJA
do i=minyr,maxyr
 z=0 !value
 zc=0 !counter
 do j=6,8
  if (perc_arr(i,j) .ne.miss_internal) then
   z=z+perc_arr(i,j)
   zc=zc+1
  endif
 enddo
 if (((frac(i,6,3)+frac(i,7,3)+frac(i,8,3))/3.0).lt.frac_thresh) then
  ts(indexID,i,3)=miss_internal
 else
  ts(indexID,i,3)=z/zc
 endif
enddo

! calculate for SON
do i=minyr,maxyr
 z=0 !value
 zc=0 !counter
 do j=9,11
  if (perc_arr(i,j) .ne.miss_internal) then
   z=z+perc_arr(i,j)
   zc=zc+1
  endif
 enddo
 if (((frac(i,9,3)+frac(i,10,3)+frac(i,11,3))/3.0).lt.frac_thresh) then
  ts(indexID,i,4)=miss_internal
 else
  ts(indexID,i,4)=z/zc
 endif
enddo

! calculate for ANNUAL
do i=minyr,maxyr
 z=0 !value
 zc=0 !counter
 do j=1,12
  if (perc_arr(i,j) .ne.miss_internal) then
   z=z+perc_arr(i,j)
   zc=zc+1
  endif
 enddo
 if (frac(i,13,3) < frac_thresh) then
  ts(indexID,i,5)=miss_internal
 else
  ts(indexID,i,5)=z/zc
 endif
enddo

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

  subroutine Tn90
!the % of time Tmin > 90th percentile (percentiles based on 61-90)

include 'indices.inc'
integer i,j,k,n
real arr(nyrs*100),perc_arr(minyr:maxyr,12)
real p90,z,z1,z2,zc
real quickpile
integer realdaysinmth

do j=1,12
 n=0
 do i=bm_minyr,bm_maxyr
  do k=1,realdaysinmth(i,j)
   if(data_in(i,j,k,3) .ne.miss_internal)then !data present
    n=n+1
    arr(n)=data_in(i,j,k,3)
   endif
  enddo
 enddo
 p90=quickpile(arr,n,90.0,miss_internal)
! p90 is now the value of the 90th percentile
 do i=minyr,maxyr
  z1=0
  z2=0
  do k=1,realdaysinmth(i,j)
   if(data_in(i,j,k,3) .ne.miss_internal)then !data present
    if(data_in(i,j,k,3) > p90)then
     z1=z1+1
    else
     z2=z2+1
    endif
   endif
  enddo
  z=miss_internal
  if(z1+z2 > 0)z=z1/(z1+z2)
  if(frac(i,j,3).lt.frac_thresh)z=miss_internal
  perc_arr(i,j)=z
 enddo
enddo

! calculate for DJF
do i=minyr,maxyr
 z=0 !value
 zc=0 !counter

 if (i /= minyr) then
  if (perc_arr(i-1,12) .ne.miss_internal) then
   z=z+perc_arr(i-1,12)
   zc=zc+1
  endif

  do j=1,2
   if (perc_arr(i,j) .ne.miss_internal) then
    z=z+perc_arr(i,j)
    zc=zc+1
   endif
  enddo

  if (((frac(i-1,12,3)+frac(i,1,3)+frac(i,2,3))/3.0).lt.frac_thresh) then
   ts(indexID,i,1)=miss_internal
  else
   ts(indexID,i,1)=z/zc
  endif
 endif
enddo

! calculate for MAM
do i=minyr,maxyr
 z=0 !value
 zc=0 !counter
 do j=3,5
  if(perc_arr(i,j) .ne.miss_internal)then
   z=z+perc_arr(i,j)
   zc=zc+1
  endif
 enddo
 if (((frac(i,3,3)+frac(i,4,3)+frac(i,5,3))/3.0).lt.frac_thresh) then
  ts(indexID,i,2)=miss_internal
 else
  ts(indexID,i,2)=z/zc
 endif
enddo

! calculate for JJA
do i=minyr,maxyr
 z=0 !value
 zc=0 !counter
 do j=6,8
  if (perc_arr(i,j) .ne.miss_internal) then
   z=z+perc_arr(i,j)
   zc=zc+1
  endif
 enddo
 if (((frac(i,6,3)+frac(i,7,3)+frac(i,8,3))/3.0).lt.frac_thresh) then
  ts(indexID,i,3)=miss_internal
 else
  ts(indexID,i,3)=z/zc
 endif
enddo

! calculate for SON
do i=minyr,maxyr
 z=0 !value
 zc=0 !counter
 do j=9,11
  if(perc_arr(i,j) .ne.miss_internal)then
   z=z+perc_arr(i,j)
   zc=zc+1
  endif
 enddo
 if (((frac(i,9,3)+frac(i,10,3)+frac(i,11,3))/3.0).lt.frac_thresh) then
  ts(indexID,i,4)=miss_internal
 else
  ts(indexID,i,4)=z/zc
 endif
enddo

! calculate for ANNUAL
do i=minyr,maxyr
 z=0 !value
 zc=0 !counter
 do j=1,12
  if (perc_arr(i,j) .ne.miss_internal) then
   z=z+perc_arr(i,j)
   zc=zc+1
  endif
 enddo
 if (frac(i,13,3) < frac_thresh)then
  ts(indexID,i,5)=miss_internal
 else
  ts(indexID,i,5)=z/zc
 endif
enddo

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

subroutine GetGrowing(i,ibegdy,ienddy,dd)
! calculate growing season

include 'indices.inc'

integer i,j,k,numdays,yr,mth
integer nc,iflagb,ibegdy,iflage,ienddy
real tmean(366),thresh,dd
integer realdaysinmth,nonmiss

thresh=DegDays_Thresh

numdays=0
nonmiss=0
yr=i
if(south)then
 mth=6
else
 mth=0
endif
do j=1,12
 mth=mth+1
 if(mth > 12)then
  mth=1
  yr=i+1
 endif
 do k=1,realdaysinmth(yr,mth)
  numdays=numdays+1
  tmean(numdays)=data_in(yr,mth,k,4)
  if(tmean(numdays).ne.miss_internal)nonmiss=nonmiss+1
 enddo
enddo

if(nonmiss.lt.frac_thresh*365)then
 ibegdy=0
 ienddy=0
 dd=0
 return
end if

! Determine the start of the Thermal Growing Season
! (i.e. T > 5 deg C for > 5 days)

nc = 0
iflagb = 0
ibegdy = 0
do k=1,numdays
 if(tmean(k) > thresh) then
  nc=nc+1
 else
  nc=0 ! start over counting the days
 endif
 if ((nc > 5).and.(iflagb.eq.0)) then ! want the first
  ibegdy = k-5 ! want the first day of the group
  iflagb = 1
 endif
enddo

! Determine the end of the Thermal Growing Season
! (i.e. T < 5 deg C for > 5 days)

nc = 0
iflage = 0
ienddy = 0
do k=numdays,numdays/2,-1 ! only look at second half of year
 if(tmean(k).lt.thresh.and.tmean(k) .ne.miss_internal)then
  nc=nc+1
 else
  nc=0 ! start over counting the days
 endif
 if (nc > 5) then ! going backwards so don't worry about iflage
  ienddy = k ! going backwards to k is the first day in the run
  iflage = 1
 endif
enddo

dd=0
if(ibegdy.ne.0.and.ienddy.ne.0)then
 do k=ibegdy,ienddy
  if(tmean(k) .ne.miss_internal)dd=dd+tmean(k)-thresh
 enddo
endif

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!end of temperature stats subs !!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!! start of precipitation stats subs !!!!!!!!!!!


subroutine prec_perc(percent)

include 'indices.inc'
integer i,j,k,n
real arr(nyrs*31*12)
real quickpile, percent
integer realdaysinmth,realdaysinssn

do i=minyr,maxyr
 do j=1,4
  if(frac_ssn(i,j,1) .ge. frac_thresh) then
   n=0
   do k=1,realdaysinssn(i,j)
    if (data_ssn(i,j,k,1) > wd_cutoff) then !rain day
     n=n+1
     arr(n)=data_ssn(i,j,k,1)
    endif
   enddo
   ts(indexID,i,j)=quickpile(arr,n,percent,miss_internal)
  else
   ts(indexID,i,j)=miss_internal
  endif
 end do
end do

! calc the annual
do i=minyr,maxyr
 if (frac(i,13,1) .ge. frac_thresh) then
  n=0
  do j=1,12
   do k=1,realdaysinmth(i,j)
    if (data_in(i,j,k,1) > wd_cutoff) then !rain day
     n=n+1
     arr(n)=data_in(i,j,k,1)
    endif
   enddo
  enddo
  ts(indexID,i,5)=quickpile(arr,n,percent,miss_internal)
 else
  ts(indexID,i,5)=miss_internal
 endif
enddo

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

subroutine prec_quant(percent)

include 'indices.inc'
integer i,j,k,m,n
real arr(nyrs*100)
real p
real quickpile, percent
integer realdaysinmth
real total_prec, above_perc, fract_above

do i=minyr,maxyr
 n=0
! calc for DJF
 if(i.eq.minyr)then
  ts(indexID,i,1)=miss_internal
 elseif (((frac(i-1,12,1)+frac(i,1,1)+frac(i,2,1))/3.0) .ge. frac_thresh) then
  do k=1,realdaysinmth(i-1,12)
   if (data_in(i-1,12,k,1) > wd_cutoff) then !rain day
    n=n+1
    arr(n)=data_in(i-1,12,k,1)
   endif
  enddo

  do j=1,2
   do k=1,realdaysinmth(i,j)
    if (data_in(i,j,k,1) > wd_cutoff) then !rain day
     n=n+1
     arr(n)=data_in(i,j,k,1)
    endif
   enddo
  enddo
  p=quickpile(arr,n,percent,miss_internal)
! p is now the value of the percentile for DJF

  total_prec=0

  do m=1,n
   total_prec = total_prec + arr(m)
  enddo

  above_perc=0

  do m=1,n
   if (arr(m) > p) then
    above_perc = above_perc + arr(m)
   endif
  enddo
  if(total_prec.gt.0.0)then
   fract_above = above_perc / total_prec
  else
   fract_above = miss_internal
  end if
  ts(indexID,i,1)=fract_above
 else
  ts(indexID,i,1)=miss_internal
 endif
 n=0

! calc for MAM
 if (((frac(i,3,1)+frac(i,4,1)+frac(i,5,1))/3.0) .ge. frac_thresh) then
  do j=3,5
   do k=1,realdaysinmth(i,j)
    if (data_in(i,j,k,1) > wd_cutoff) then !rain day
     n=n+1
     arr(n)=data_in(i,j,k,1)
    endif
   enddo
  enddo
  p=quickpile(arr,n,percent,miss_internal)
! p is now the value of the percentile for MAM

  total_prec=0

  do m=1,n
   total_prec = total_prec + arr(m)
  enddo

  above_perc=0

  do m=1,n
   if (arr(m) > p) then
    above_perc = above_perc + arr(m)
   endif
  enddo

  if(total_prec.gt.0.0)then
   fract_above = above_perc / total_prec
  else
   fract_above = miss_internal
  end if
  ts(indexID,i,2)=fract_above
 else
  ts(indexID,i,2)=miss_internal
 endif

 n=0

! calc for JJA
 if (((frac(i,6,1)+frac(i,7,1)+frac(i,8,1))/3.0) .ge. frac_thresh) then
  do j=6,8
   do k=1,realdaysinmth(i,j)
    if (data_in(i,j,k,1) > wd_cutoff) then !rain day
     n=n+1
     arr(n)=data_in(i,j,k,1)
    endif
   enddo
  enddo
  p=quickpile(arr,n,percent,miss_internal)
! p is now the value of the percentile for JJA

  total_prec=0

  do m=1,n
  total_prec = total_prec + arr(m)
  enddo

  above_perc=0

  do m=1,n
   if (arr(m) > p) then
    above_perc = above_perc + arr(m)
   endif
  enddo

  if(total_prec.gt.0.0)then
   fract_above = above_perc / total_prec
  else
   fract_above = miss_internal
  end if
  ts(indexID,i,3)=fract_above
 else
  ts(indexID,i,3)=miss_internal
 endif

 n=0

! calc for SON
 if (((frac(i,9,1)+frac(i,10,1)+frac(i,11,1))/3.0) .ge. frac_thresh) then
  do j=9,11
   do k=1,realdaysinmth(i,j)
    if (data_in(i,j,k,1) > wd_cutoff) then !rain day
     n=n+1
     arr(n)=data_in(i,j,k,1)
    endif
   enddo
  enddo
  p=quickpile(arr,n,percent,miss_internal)
! p is now the value of the percentile for SON

  total_prec=0

  do m=1,n
   total_prec = total_prec + arr(m)
  enddo

  above_perc=0

  do m=1,n
   if (arr(m) > p) then
    above_perc = above_perc + arr(m)
   endif
  enddo

  if(total_prec.gt.0.0)then
   fract_above = above_perc / total_prec
  else
   fract_above = miss_internal
  end if
  ts(indexID,i,4)=fract_above
 else
  ts(indexID,i,4)=miss_internal
 endif
enddo

! calc the annual
do i=minyr,maxyr
 n=0
 if (frac(i,13,1) .ge. frac_thresh) then
  do j=1,12
   do k=1,realdaysinmth(i,j)
    if (data_in(i,j,k,1) > wd_cutoff) then !rain day
     n=n+1
     arr(n)=data_in(i,j,k,1)
    endif
   enddo
  enddo
  p=quickpile(arr,n,percent,miss_internal)
! p is now the value of the percentile for the annual data

  total_prec=0

  do m=1,n
   total_prec = total_prec + arr(m)
  enddo

  above_perc=0

  do m=1,n
   if (arr(m) > p) then
    above_perc = above_perc + arr(m)
   endif
  enddo

  if(total_prec.gt.0.0)then
   fract_above = above_perc / total_prec
  else
   fract_above = miss_internal
  end if
  ts(indexID,i,5)=fract_above
 else
  ts(indexID,i,5)=miss_internal
 endif
enddo

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

subroutine Above_Thresh(ivar,thresh)
! this index calculates the number of days with var >= thresh

include 'indices.inc'
integer i,j,k,ivar,realdaysinssn
integer muse(12)
real dnum,thresh
integer realdaysinmth


! seasonal
do j=1,4
 do i=minyr,maxyr
  if(frac_ssn(i,j,ivar) .ge. frac_thresh)then
   dnum=0
   do k=1,realdaysinssn(i,j)
    if(data_ssn(i,j,k,ivar).ne.miss_internal.and.data_ssn(i,j,k,ivar).ge.thresh)then
     dnum=dnum+1
    endif
   enddo
   ts(indexID,i,j)=dnum
  end if
 enddo
enddo

! annual
do j=1,12
 muse(j)=0
enddo
do i=minyr,maxyr
 if(frac(i,13,ivar) .ge. frac_thresh)then
  dnum=0
  do j=1,12
   do k=1,realdaysinmth(i,j)
    if(data_in(i,j,k,ivar) .ne.miss_internal)then !data present
     if(data_in(i,j,k,ivar).ge.thresh)then
      dnum=dnum+1
      muse(j)=1
     endif
    endif
   enddo
  enddo
  ts(indexID,i,5)=dnum
 end if
enddo

call normalize(indexID,muse,ivar)

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

subroutine cdd  !consecutive dry days

include 'indices.inc'
integer i,j,k
integer nc,ncmax
integer realdaysinmth,realdaysinssn

! seasonal 
do i=minyr,maxyr
 do j=1,4
  nc=0
  ncmax=0
  if(frac_ssn(i,j,1) .ge. frac_thresh)then ! must have 80% of days present
   do k=1,realdaysinssn(i,j)
    if(data_ssn(i,j,k,1) < wd_cutoff.and.data_ssn(i,j,k,1) .ne.miss_internal)then ! it is a dry day
     nc=nc+1
    else
     nc=0
    end if
    ncmax=max(ncmax,nc)
   enddo
   ts(indexID,i,j)=ncmax
  endif
 enddo
enddo

! annual
do i=minyr,maxyr
 nc=0
 ncmax=0
 if(frac(i,13,1) .ge. frac_thresh)then ! must have 80% of days present
  do j=1,12 !through months
   do k=1,realdaysinmth(i,j)
    if(data_in(i,j,k,1) < wd_cutoff.and.data_in(i,j,k,1) .ne.miss_internal)then ! it is a dry day
     nc=nc+1
    else
     nc=0
    end if
    ncmax=max(ncmax,nc)
   enddo
  enddo
  ts(indexID,i,5)=ncmax
 endif
enddo

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

subroutine cwd  !consecutive wet days

include 'indices.inc'
integer i,j,k
integer nc,ncmax
integer realdaysinmth,realdaysinssn

! seasonal 
do i=minyr,maxyr
 do j=1,4
  nc=0
  ncmax=0
  if(frac_ssn(i,j,1) .ge. frac_thresh)then ! must have 80% of days present
   do k=1,realdaysinssn(i,j)
    if(data_ssn(i,j,k,1) >= wd_cutoff.and.data_ssn(i,j,k,1) .ne.miss_internal)then ! it is a wet day
     nc=nc+1
    else
     nc=0
    end if
    ncmax=max(ncmax,nc)
   enddo
   ts(indexID,i,j)=ncmax
  endif
 enddo
enddo

! annual
do i=minyr,maxyr
 nc=0
 ncmax=0
 if(frac(i,13,1) .ge. frac_thresh)then ! must have 80% of days present
  do j=1,12 !through months
   do k=1,realdaysinmth(i,j)
    if(data_in(i,j,k,1) >= wd_cutoff.and.data_in(i,j,k,1) .ne.miss_internal)then ! it is a wet day
     nc=nc+1
    else
     nc=0
    end if
    ncmax=max(ncmax,nc)
   enddo
  enddo
  ts(indexID,i,5)=ncmax
 endif
enddo

return
end


!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

subroutine prec_pww(pww_arr)
!Mean wet-day persistence (Pww)
include 'indices.inc'
integer i,j,k,m,n
real arr(400)
real pair11,pair10,pww
integer realdaysinmth
real pww_arr(minyr:maxyr,5)

do i=minyr,maxyr
 do j=1,5
  pww_arr(i,j)=miss_internal
 enddo
enddo

do i=minyr,maxyr
 n=0
! calc for DJF
 if(i.eq.minyr)then
  ts(indexID,i,1)=miss_internal
  pww_arr(i,1)=miss_internal
 elseif (((frac(i-1,12,1)+frac(i,1,1)+frac(i,2,1))/3.0) .ge. frac_thresh) then
  do k=1,realdaysinmth(i-1,12)
   if (data_in(i-1,12,k,1) .ne.miss_internal) then !data present
    n=n+1
    arr(n)=data_in(i-1,12,k,1)
   endif
  enddo

  do j=1,2
   do k=1,realdaysinmth(i,j)
    if (data_in(i,j,k,1) .ne.miss_internal) then !data present
     n=n+1
     arr(n)=data_in(i,j,k,1)
    endif
   enddo
  enddo
  if (data_in(i,3,1,1) .ne.miss_internal) then !data present
   n=n+1
   arr(n)=data_in(i,3,1,1)
  endif

  pair11=0
  do m=2,n
   if (arr(m-1) > wd_cutoff .and. arr(m) > wd_cutoff) then
    pair11 = pair11 + 1.0
   endif
  enddo

  pair10=0
  do m=2,n
   if (arr(m-1) > wd_cutoff .and. arr(m) <= wd_cutoff) then
    pair10 = pair10 + 1.0
   endif
  enddo

  if((pair11 + pair10).gt.0.0)then
   pww = pair11/(pair11 + pair10)
  else
   pww=0
  end if

  ts(indexID,i,1)=pww
  pww_arr(i,1)=pww
 else
  ts(indexID,i,1)=miss_internal
  pww_arr(i,1)=miss_internal
 endif

 n=0
! calc for MAM
 if(((frac(i,3,1)+frac(i,4,1)+frac(i,5,1))/3.0) .ge. frac_thresh) then
  do j=3,5
   do k=1,realdaysinmth(i,j)
    if(data_in(i,j,k,1) .ne.miss_internal)then !data present
     n=n+1
     arr(n)=data_in(i,j,k,1)
    endif
   enddo
  enddo

  if (data_in(i,6,1,1) .ne.miss_internal) then !data present
   n=n+1
   arr(n)=data_in(i,6,1,1)
  endif

  pair11=0
  do m=2,n
   if (arr(m-1) > wd_cutoff .and. arr(m) > wd_cutoff) then
    pair11 = pair11 + 1.0
   endif
  enddo

  pair10=0
  do m=2,n
   if (arr(m-1) > wd_cutoff .and. arr(m) <= wd_cutoff) then
    pair10 = pair10 + 1.0
   endif
  enddo

  if((pair11 + pair10).gt.0.0)then
   pww = pair11/(pair11 + pair10)
  else
   pww=0
  end if
  ts(indexID,i,2)=pww
  pww_arr(i,2)=pww
 else
  ts(indexID,i,2)=miss_internal
  pww_arr(i,2)=miss_internal
 endif

 n=0

! calc for JJA
 if(((frac(i,6,1)+frac(i,7,1)+frac(i,8,1))/3.0) .ge. frac_thresh) then
  do j=6,8
   do k=1,realdaysinmth(i,j)
    if (data_in(i,j,k,1) .ne.miss_internal) then !data present
     n=n+1
     arr(n)=data_in(i,j,k,1)
    endif
   enddo
  enddo

  if (data_in(i,9,1,1) .ne.miss_internal) then !data present
   n=n+1
   arr(n)=data_in(i,9,1,1)
  endif

  pair11=0
  do m=2,n
   if (arr(m-1) > wd_cutoff .and. arr(m) > wd_cutoff) then
    pair11 = pair11 + 1.0
   endif
  enddo

  pair10=0
  do m=2,n
   if (arr(m-1) > wd_cutoff .and. arr(m) <= wd_cutoff) then
    pair10 = pair10 + 1.0
   endif
  enddo

  if((pair11 + pair10).gt.0.0)then
   pww = pair11/(pair11 + pair10)
  else
   pww=0
  end if
  ts(indexID,i,3)=pww
  pww_arr(i,3)=pww
 else
  ts(indexID,i,3)=miss_internal
  pww_arr(i,3)=miss_internal
 endif

 n=0

! calc for SON
 if(((frac(i,9,1)+frac(i,10,1)+frac(i,11,1))/3.0) .ge. frac_thresh) then
  do j=9,11
   do k=1,realdaysinmth(i,j)
    if (data_in(i,j,k,1) .ne.miss_internal) then !data present
     n=n+1
     arr(n)=data_in(i,j,k,1)
    endif
   enddo
  enddo

  if (data_in(i,12,1,1) .ne.miss_internal) then !data present
   n=n+1
   arr(n)=data_in(i,12,1,1)
  endif

  pair11=0
  do m=2,n
   if (arr(m-1) > wd_cutoff .and. arr(m) > wd_cutoff) then
    pair11 = pair11 + 1.0
   endif
  enddo

  pair10=0
  do m=2,n
   if (arr(m-1) > wd_cutoff .and. arr(m) <= wd_cutoff) then
    pair10 = pair10 + 1.0
   endif
  enddo

  if((pair11 + pair10).gt.0.0)then
   pww = pair11/(pair11 + pair10)
  else
   pww=0
  end if
  ts(indexID,i,4)=pww
  pww_arr(i,4)=pww
 else
  ts(indexID,i,4)=miss_internal
  pww_arr(i,4)=miss_internal
 endif
enddo

! calc the annual
do i=minyr,maxyr
n=0
 if(frac(i,13,1) .ge. frac_thresh) then
  do j=1,12
   do k=1,realdaysinmth(i,j)
    if (data_in(i,j,k,1) .ne.miss_internal)then !data present
     n=n+1
     arr(n)=data_in(i,j,k,1)
    endif
   enddo
  enddo

  if(i.ne.maxyr)then
   if(data_in(i+1,1,1,1) .ne.miss_internal) then !data present
    n=n+1
    arr(n)=data_in(i+1,1,1,1)
   endif
  endif

  pair11=0
  do m=2,n
   if (arr(m-1) > wd_cutoff .and. arr(m) > wd_cutoff) then
    pair11 = pair11 + 1.0
   endif
  enddo

  pair10=0
  do m=2,n
   if (arr(m-1) > wd_cutoff .and. arr(m) <= wd_cutoff) then
    pair10 = pair10 + 1.0
   endif
  enddo

  if((pair11 + pair10).gt.0.0)then
   pww = pair11/(pair11 + pair10)
  else
   pww=0
  end if
  ts(indexID,i,5)=pww
  pww_arr(i,5)=pww
 else
  ts(indexID,i,5)=miss_internal
  pww_arr(i,5)=miss_internal
 endif
enddo

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

subroutine prec_pdd(pdd_arr) !Mean dry-day persistence (Pdd)
include 'indices.inc'
integer i,j,k,m,n
real arr(400)
real pair11,pair10,pdd
integer realdaysinmth
real pdd_arr(minyr:maxyr,5)

do i=minyr,maxyr
 do j=1,5
  pdd_arr(i,j)=miss_internal
 enddo
enddo

do i=minyr,maxyr
 n=0
! calc for DJF
 if(i.eq.minyr)then
  ts(indexID,i,1)=miss_internal
  pdd_arr(i,1)=miss_internal
 elseif (((frac(i-1,12,1)+frac(i,1,1)+frac(i,2,1))/3.0) .ge. frac_thresh) then
  do k=1,realdaysinmth(i-1,12)
   if (data_in(i-1,12,k,1) .ne.miss_internal) then !data present
    n=n+1
    arr(n)=data_in(i-1,12,k,1)
   endif
  enddo

  do j=1,2
   do k=1,realdaysinmth(i,j)
    if (data_in(i,j,k,1) .ne.miss_internal) then !data present
     n=n+1
     arr(n)=data_in(i,j,k,1)
    endif
   enddo
  enddo
  if (data_in(i,3,1,1) .ne.miss_internal) then !data present
   n=n+1
   arr(n)=data_in(i,3,1,1)
  endif

  pair11=0
  do m=2,n
   if (arr(m-1) <= wd_cutoff .and. arr(m) <= wd_cutoff) then
    pair11 = pair11 + 1.0
   endif
  enddo

  pair10=0
  do m=2,n
   if (arr(m-1) <= wd_cutoff .and. arr(m) > wd_cutoff) then
    pair10 = pair10 + 1.0
   endif
  enddo

  if((pair11 + pair10).gt.0.0)then
   pdd = pair11/(pair11 + pair10)
  else
   pdd=0
  end if
  ts(indexID,i,1)=pdd
  pdd_arr(i,1)=pdd
 else
  ts(indexID,i,1)=miss_internal
  pdd_arr(i,1)=miss_internal
 endif

 n=0
! calc for MAM
 if(((frac(i,3,1)+frac(i,4,1)+frac(i,5,1))/3.0) .ge. frac_thresh) then
  do j=3,5
   do k=1,realdaysinmth(i,j)
    if(data_in(i,j,k,1) .ne.miss_internal)then !data present
     n=n+1
     arr(n)=data_in(i,j,k,1)
    endif
   enddo
  enddo

  if (data_in(i,6,1,1) .ne.miss_internal) then !data present
   n=n+1
   arr(n)=data_in(i,6,1,1)
  endif

  pair11=0
  do m=2,n
   if (arr(m-1) <= wd_cutoff .and. arr(m) <= wd_cutoff) then
    pair11 = pair11 + 1.0
   endif
  enddo

  pair10=0
  do m=2,n
   if (arr(m-1) <= wd_cutoff .and. arr(m) > wd_cutoff) then
    pair10 = pair10 + 1.0
   endif
  enddo
  if((pair11 + pair10).gt.0.0)then
   pdd = pair11/(pair11 + pair10)
  else
   pdd=0
  end if
  ts(indexID,i,2)=pdd
  pdd_arr(i,2)=pdd
 else
  ts(indexID,i,2)=miss_internal
  pdd_arr(i,2)=miss_internal
 endif

 n=0

! calc for JJA
 if(((frac(i,6,1)+frac(i,7,1)+frac(i,8,1))/3.0) .ge. frac_thresh) then
  do j=6,8
   do k=1,realdaysinmth(i,j)
    if (data_in(i,j,k,1) .ne.miss_internal) then !data present
     n=n+1
     arr(n)=data_in(i,j,k,1)
    endif
   enddo
  enddo

  if (data_in(i,9,1,1) .ne.miss_internal) then !data present
   n=n+1
   arr(n)=data_in(i,9,1,1)
  endif

  pair11=0
  do m=2,n
   if (arr(m-1) <= wd_cutoff .and. arr(m) <= wd_cutoff) then
    pair11 = pair11 + 1.0
   endif
  enddo

  pair10=0
  do m=2,n
   if (arr(m-1) <= wd_cutoff .and. arr(m) > wd_cutoff) then
    pair10 = pair10 + 1.0
   endif
  enddo
  if((pair11 + pair10).gt.0.0)then
   pdd = pair11/(pair11 + pair10)
  else
   pdd=0
  end if
  ts(indexID,i,3)=pdd
  pdd_arr(i,3)=pdd
 else
  ts(indexID,i,3)=miss_internal
  pdd_arr(i,3)=miss_internal
 endif

 n=0

! calc for SON
 if(((frac(i,9,1)+frac(i,10,1)+frac(i,11,1))/3.0) .ge. frac_thresh) then
  do j=9,11
   do k=1,realdaysinmth(i,j)
    if (data_in(i,j,k,1) .ne.miss_internal) then !data present
     n=n+1
     arr(n)=data_in(i,j,k,1)
    endif
   enddo
  enddo

  if (data_in(i,12,1,1) .ne.miss_internal) then !data present
   n=n+1
   arr(n)=data_in(i,12,1,1)
  endif

  pair11=0
  do m=2,n
   if (arr(m-1) <= wd_cutoff .and. arr(m) <= wd_cutoff) then
    pair11 = pair11 + 1.0
   endif
  enddo

  pair10=0
  do m=2,n
   if (arr(m-1) <= wd_cutoff .and. arr(m) > wd_cutoff) then
    pair10 = pair10 + 1.0
   endif
  enddo
  if((pair11 + pair10).gt.0.0)then
   pdd = pair11/(pair11 + pair10)
  else
   pdd=0
  end if
  ts(indexID,i,4)=pdd
  pdd_arr(i,4)=pdd
 else
  ts(indexID,i,4)=miss_internal
  pdd_arr(i,4)=miss_internal
 endif
enddo

! calc the annual
do i=minyr,maxyr
n=0
 if(frac(i,13,1) .ge. frac_thresh) then
  do j=1,12
   do k=1,realdaysinmth(i,j)
    if (data_in(i,j,k,1) .ne.miss_internal)then !data present
     n=n+1
     arr(n)=data_in(i,j,k,1)
    endif
   enddo
  enddo

  if(i.ne.maxyr)then
   if(data_in(i+1,1,1,1) .ne.miss_internal) then !data present
    n=n+1
    arr(n)=data_in(i+1,1,1,1)
   endif
  endif

  pair11=0
  do m=2,n
   if (arr(m-1) <= wd_cutoff .and. arr(m) <= wd_cutoff) then
    pair11 = pair11 + 1.0
   endif
  enddo

  pair10=0
  do m=2,n
   if (arr(m-1) <= wd_cutoff .and. arr(m) > wd_cutoff) then
    pair10 = pair10 + 1.0
   endif
  enddo
  if((pair11 + pair10).gt.0.0)then
   pdd = pair11/(pair11 + pair10)
  else
   pdd=0
  end if
  ts(indexID,i,5)=pdd
  pdd_arr(i,5)=pdd
 else
  ts(indexID,i,5)=miss_internal
  pdd_arr(i,5)=miss_internal
 endif
enddo

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

! calculate the persistence correlation r

subroutine persist_corr(pww_arr,pdd_arr)
include 'indices.inc'

real pww_arr(minyr:maxyr,5),pdd_arr(minyr:maxyr,5)
real r
integer i

do i=minyr,maxyr

 if (pww_arr(i,1) /= miss_internal .and. pdd_arr(i,1) /= miss_internal) then
  r = pww_arr(i,1) - (1 - pdd_arr(i,1))
  ts(indexID,i,1)=r
 else
  ts(indexID,i,1)=miss_internal
 endif

 if (pww_arr(i,2) /= miss_internal .and. pdd_arr(i,2) /= miss_internal) then
  r = pww_arr(i,2) - (1 - pdd_arr(i,2))
  ts(indexID,i,2)=r
 else
  ts(indexID,i,2)=miss_internal
 endif
 if (pww_arr(i,3) /= miss_internal .and. pdd_arr(i,3) /= miss_internal) then
  r = pww_arr(i,3) - (1 - pdd_arr(i,3))
  ts(indexID,i,3)=r
 else
  ts(indexID,i,3)=miss_internal
 endif
 if (pww_arr(i,4) /= miss_internal .and. pdd_arr(i,4) /= miss_internal) then
  r = pww_arr(i,4) - (1 - pdd_arr(i,4))
  ts(indexID,i,4)=r
 else
  ts(indexID,i,4)=miss_internal
 endif
 if (pww_arr(i,5) /= miss_internal .and. pdd_arr(i,5) /= miss_internal) then
  r = pww_arr(i,5) - (1 - pdd_arr(i,5))
  ts(indexID,i,5)=r
 else
  ts(indexID,i,5)=miss_internal
 endif

enddo

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

subroutine r3_5_10d(idays)
! creates 3, 5, or 10-day rainfall total

include 'indices.inc'
integer i,j,yr,mth,day
real arr(366)
integer idays,n,issn,realdaysinssn
real bigtot,tot
integer realdaysinmth

! seasonal
do yr=minyr,maxyr
 do issn=1,4
  if(frac_ssn(yr,issn,1).ge.frac_thresh)then !requires 80% of data to be present
   n=0
   do day=1,realdaysinssn(yr,issn)
    n=n+1
    arr(n)=data_ssn(yr,issn,day,1)
   end do
   bigtot=0
   do i=1,n-idays+1
    tot=0
    do j=1,idays
     if(arr(i+j-1).ne.miss_internal)tot=tot+arr(i+j-1)
    end do
    if(tot.gt.bigtot)bigtot=tot
   end do
   ts(indexID,yr,issn)=bigtot
  endif
 end do
end do

! annual
do yr=minyr,maxyr
 if(frac(yr,13,1).ge.frac_thresh)then !requires 80% of data to be present
  n=0
  do mth=1,12
   do day=1,realdaysinmth(yr,mth)
    n=n+1
    arr(n)=data_in(yr,mth,day,1)
   end do
  end do
  bigtot=0
  do i=1,n-idays+1
   tot=0
   do j=1,idays
    if(arr(i+j-1).ne.miss_internal)tot=tot+arr(i+j-1)
   end do
   if(tot.gt.bigtot)bigtot=tot
  end do
  ts(indexID,yr,5)=bigtot
 endif
end do

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

subroutine sdii
! this index calculates the Simple Daily Intensity Index

include 'indices.inc'
integer i,j,k,dnum,issn
real anntot
integer realdaysinmth,realdaysinssn

! seasonal 
do i=minyr,maxyr
 do issn=1,4
  anntot=0
  dnum=0
  do k=1,realdaysinssn(i,issn)
   if(data_ssn(i,issn,k,1).ne.miss_internal)then !data present
    if(data_ssn(i,issn,k,1).ge.wd_cutoff)then
     anntot=anntot+data_ssn(i,issn,k,1)
     dnum=dnum+1
    end if
   endif
  enddo
  if(dnum > 0)then !don't do it if no days with precip
   if(frac_ssn(i,issn,1).ge.frac_thresh)then
    ts(indexID,i,issn)=anntot/real(dnum)
   endif
  endif
 enddo
enddo

! annual
do i=minyr,maxyr
 anntot=0
 dnum=0
 do j=1,12
  do k=1,realdaysinmth(i,j)
   if(data_in(i,j,k,1).ne.miss_internal)then !data present
    if(data_in(i,j,k,1).ge.wd_cutoff)then
     anntot=anntot+data_in(i,j,k,1)
     dnum=dnum+1
    end if
   endif
  enddo
 enddo
 if(dnum > 0)then !don't do it if no days with precip
  if(frac(i,13,1).ge.frac_thresh)then
   ts(indexID,i,5)=anntot/real(dnum)
  endif
 endif
enddo

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

subroutine RThreshT(pile)

include 'indices.inc'
integer i,j,k,n
real arr(nyrs*366),total,above95,p95
real quickpile,pile
integer realdaysinmth,realdaysinssn

! seasonal 
do j=1,4
 n = 0
 do i=bm_minyr,bm_maxyr
  do k=1,realdaysinssn(i,j)
   if(data_ssn(i,j,k,1).ne.miss_internal.and.data_ssn(i,j,k,1).gt.wd_cutoff)then
    n=n+1
    arr(n)=data_ssn(i,j,k,1)
   endif
  enddo
 enddo

 p95=quickpile(arr,n,pile,miss_internal)

 do i=minyr,maxyr
  total = 0
  above95 = 0
  do k=1,realdaysinssn(i,j)
   if(data_ssn(i,j,k,1) .ne.miss_internal)then !data_in present
    total = total + data_ssn(i,j,k,1)
    if(data_ssn(i,j,k,1).ge.p95)then
     above95 = above95+data_ssn(i,j,k,1)
    endif
   endif
  enddo

  if (total > 0.and.frac_ssn(i,j,1).ge.frac_thresh) then
   ts(indexID,i,j) = above95/total
  endif

 enddo
enddo

! annual
n = 0
do i=bm_minyr,bm_maxyr
 do j=1,12
  do k=1,realdaysinmth(i,j)
   if(data_in(i,j,k,1).ne.miss_internal.and.data_in(i,j,k,1).gt.wd_cutoff)then
    n=n+1
    arr(n)=data_in(i,j,k,1)
   endif
  enddo
 enddo
enddo

p95=quickpile(arr,n,pile,miss_internal)

do i=minyr,maxyr
 total = 0
 above95 = 0
 do j=1,12
  do k=1,realdaysinmth(i,j)
   if(data_in(i,j,k,1) .ne.miss_internal)then !data_in present
    total = total + data_in(i,j,k,1)
    if(data_in(i,j,k,1).ge.p95)then
     above95 = above95+data_in(i,j,k,1)
    endif
   endif
  enddo
 enddo

 if (total > 0.and.frac(i,13,1).ge.frac_thresh) then
  ts(indexID,i,5) = above95/total
 endif

enddo

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

subroutine RThreshN(pile)

include 'indices.inc'
integer i,j,k,n,extreme,muse(12)
real arr(nyrs*12*31),p95
real quickpile,pile
integer issn
integer realdaysinmth,realdaysinssn

! seasonal

do issn=1,4
 n = 0
 do i=bm_minyr,bm_maxyr
  do k=1,realdaysinssn(i,issn)
   if(data_ssn(i,issn,k,1).ne.miss_internal.and.data_ssn(i,issn,k,1).gt.wd_cutoff)then
    n=n+1
    arr(n)=data_ssn(i,issn,k,1)
   endif
  enddo
 enddo

 p95=quickpile(arr,n,pile,miss_internal)

 do i=minyr,maxyr
  extreme=0
  do k=1,realdaysinssn(i,issn)
   if(data_ssn(i,issn,k,1) .ne.miss_internal)then
    if(data_ssn(i,issn,k,1).ge.p95)then
     extreme=extreme+1
    endif
   endif
  enddo
  if(frac_ssn(i,issn,1).gt.frac_thresh)ts(indexID,i,issn) = extreme
 enddo

enddo

! annual
n = 0
do i=bm_minyr,bm_maxyr
 do j=1,12
  do k=1,realdaysinmth(i,j)
   if(data_in(i,j,k,1).ne.miss_internal.and.data_in(i,j,k,1).gt.wd_cutoff)then
    n=n+1
    arr(n)=data_in(i,j,k,1)
   endif
  enddo
 enddo
enddo

p95=quickpile(arr,n,pile,miss_internal)

do j=1,12
 muse(j)=0
end do
do i=minyr,maxyr
 if(frac(i,13,1).gt.frac_thresh)then
  extreme=0
  do j=1,12
   do k=1,realdaysinmth(i,j)
    if(data_in(i,j,k,1) .ne.miss_internal)then
     if(data_in(i,j,k,1).ge.p95)then
      extreme=extreme+1
      muse(j)=1
     endif
    endif
   enddo
  enddo
  ts(indexID,i,5) = extreme
 end if
end do

call normalize(indexID,muse,1)

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

!Statistics of wet spell lengths (days)

subroutine wet_spell_stats()
include 'indices.inc'
real p,quickpile,percent,mean,sd,total
integer i,j,k,m
integer crd,ioccur
integer realdaysinmth
real arr(nyrs*100)
real ts_mean(minyr:maxyr,5)
real ts_50perc(minyr:maxyr,5)
real ts_sd(minyr:maxyr,5)

logical frac_data

percent = 50.0

do i=minyr,maxyr
 crd=0
 ioccur = 0

! calc for DJF
 if(i.eq.minyr)then
  frac_data = .false.
 elseif (((frac(i-1,12,1)+frac(i,1,1)+frac(i,2,1))/3.0) .ge. frac_thresh) then
  do k=1,realdaysinmth(i-1,12)
!test for a consecutive rain day
   if (data_in(i-1,12,k,1) > wd_cutoff) then
    crd=crd+1
   else
    if (crd > 0) then
     ioccur = ioccur + 1
     arr(ioccur) = crd
    endif
    crd = 0
   endif
  enddo

  do j=1,2
   do k=1,realdaysinmth(i,j)
!test for a consecutive rain day
    if (data_in(i,j,k,1) > wd_cutoff) then
     crd=crd+1
    else
     if (crd > 0) then
      ioccur = ioccur + 1
      arr(ioccur) = crd
     endif
     crd = 0
    endif
   enddo
  enddo
    if (crd > 0) then
    ioccur = ioccur + 1
    arr(ioccur) = crd
  endif
  frac_data = .true.
 else
  frac_data = .false.
 endif

! calculate the mean for DJF
 if (frac_data) then
  total = 0
  do m=1,ioccur
   total = total + arr(m)
  enddo
  mean = total
  if(ioccur.gt.1)mean = total / real(ioccur)
  ts_mean(i,1)=mean
 else
  ts_mean(i,1)=miss_internal
 endif

! calculate the 50th percentile for DJF
 if (frac_data) then
  p=quickpile(arr,ioccur,percent,miss_internal)
  ts_50perc(i,1)=p
 else
  ts_50perc(i,1)=miss_internal
 endif

! calculate the standard deviation for DJF
 if (frac_data) then
  total = 0
  do m=1,ioccur
   total = total + ((arr(m) - mean)*(arr(m) - mean))
  enddo
  if(ioccur.gt.1)total = total / real(ioccur-1.0)
  sd = sqrt(total)
  ts_sd(i,1)=sd
 else
  ts_sd(i,1)=miss_internal
 endif

 crd=0
 ioccur=0

! calc for MAM
 if (((frac(i,3,1)+frac(i,4,1)+frac(i,5,1))/3.0) .ge. frac_thresh) then
  do j=3,5
   do k=1,realdaysinmth(i,j)
    if (data_in(i,j,k,1) > wd_cutoff) then !rain day
     crd=crd+1
    else
     if (crd > 0) then
      ioccur = ioccur + 1
      arr(ioccur) = crd
     endif
     crd = 0
    endif
   enddo
  enddo
    if (crd > 0) then
    ioccur = ioccur + 1
    arr(ioccur) = crd
  endif
  frac_data = .true.
 else
  frac_data = .false.
 endif

! calculate the mean for MAM
 if (frac_data) then
  total = 0
  do m=1,ioccur
   total = total + arr(m)
  enddo
  mean = total
  if(ioccur.gt.1)mean = total / real(ioccur)
  ts_mean(i,2)=mean
 else
  ts_mean(i,2)=miss_internal
 endif

! calculate the 50th percentile for MAM
 if (frac_data) then
  p=quickpile(arr,ioccur,percent,miss_internal)
  ts_50perc(i,2)=p
 else
  ts_50perc(i,2)=miss_internal
 endif

! calculate the standard deviation for MAM
 if (frac_data) then
  total = 0
  do m=1,ioccur
   total = total + ((arr(m) - mean)*(arr(m) - mean))
  enddo
  if(ioccur.gt.1)total = total / real(ioccur-1.0)
  sd = sqrt(total)
  ts_sd(i,2)=sd
 else
  ts_sd(i,2)=miss_internal
 endif

 crd=0
 ioccur=0

! calc for JJA
 if (((frac(i,6,1)+frac(i,7,1)+frac(i,8,1))/3.0) .ge. frac_thresh) then
  do j=6,8
   do k=1,realdaysinmth(i,j)
    if (data_in(i,j,k,1) > wd_cutoff) then !rain day
     crd=crd+1
    else
     if (crd > 0) then
      ioccur = ioccur + 1
      arr(ioccur) = crd
     endif
     crd = 0
    endif
   enddo
  enddo
    if (crd > 0) then
    ioccur = ioccur + 1
    arr(ioccur) = crd
  endif
  frac_data = .true.
 else
  frac_data = .false.
 endif

! calculate the mean for JJA
 if (frac_data) then
  total = 0
  do m=1,ioccur
   total = total + arr(m)
  enddo
  mean = total
  if(ioccur.gt.1)mean = total / real(ioccur)
  ts_mean(i,3)=mean
 else
  ts_mean(i,3)=miss_internal
 endif

! calculate the 50th percentile for JJA
 if (frac_data) then
  p=quickpile(arr,ioccur,percent,miss_internal)
  ts_50perc(i,3)=p
 else
  ts_50perc(i,3)=miss_internal
 endif

! calculate the standard deviation for JJA
 if (frac_data) then
  total = 0
  do m=1,ioccur
   total = total + ((arr(m) - mean)*(arr(m) - mean))
  enddo
  if(ioccur.gt.1)total = total / real(ioccur-1.0)
  sd = sqrt(total)
  ts_sd(i,3)=sd
 else
  ts_sd(i,3)=miss_internal
 endif

 crd=0
 ioccur=0

! calc for SON
 if (((frac(i,9,1)+frac(i,10,1)+frac(i,11,1))/3.0) .ge. frac_thresh) then
  do j=9,11
   do k=1,realdaysinmth(i,j)
    if (data_in(i,j,k,1) > wd_cutoff) then !rain day
     crd=crd+1
    else
     if (crd > 0) then
      ioccur = ioccur + 1
      arr(ioccur) = crd
     endif
     crd = 0
    endif
   enddo
  enddo
    if (crd > 0) then
    ioccur = ioccur + 1
    arr(ioccur) = crd
  endif
  frac_data = .true.
 else
  frac_data = .false.
 endif

! calculate the mean for SON
 if (frac_data) then
  total = 0
  do m=1,ioccur
   total = total + arr(m)
  enddo
  mean = total
  if(ioccur.gt.1)mean = total / real(ioccur)
  ts_mean(i,4)=mean
 else
  ts_mean(i,4)=miss_internal
 endif

! calculate the 50th percentile for SON
 if (frac_data) then
  p=quickpile(arr,ioccur,percent,miss_internal)
  ts_50perc(i,4)=p
 else
  ts_50perc(i,4)=miss_internal
 endif

! calculate the standard deviation for SON
 if (frac_data) then
  total = 0
  do m=1,ioccur
   total = total + ((arr(m) - mean)*(arr(m) - mean))
  enddo
  if(ioccur.gt.1)total = total / real(ioccur-1.0)
  sd = sqrt(total)
  ts_sd(i,4)=sd
 else
  ts_sd(i,4)=miss_internal
 endif

 crd=0
 ioccur=0

enddo

! calc the annual
do i=minyr,maxyr
 ioccur=0
 crd=0
 if (frac(i,13,1) .ge. frac_thresh) then
  do j=1,12
   do k=1,realdaysinmth(i,j)
    if (data_in(i,j,k,1) > wd_cutoff) then !rain day
     crd=crd+1
    else
     if (crd > 0) then
      ioccur = ioccur + 1
      arr(ioccur) = crd
     endif
     crd = 0
    endif
   enddo
  enddo
    if (crd > 0) then
    ioccur = ioccur + 1
    arr(ioccur) = crd
  endif
  frac_data = .true.
 else
  frac_data = .false.
 endif

! calculate the mean for ANNUAL
 if (frac_data) then
  total = 0
  do m=1,ioccur
   total = total + arr(m)
  enddo
  mean = total
  if(ioccur.gt.1)mean = total / real(ioccur)
  ts_mean(i,5)=mean
 else
  ts_mean(i,5)=miss_internal
 endif

! calculate the 50th percentile for ANNUAL
 if (frac_data) then
  p=quickpile(arr,ioccur,percent,miss_internal)
  ts_50perc(i,5)=p
 else
  ts_50perc(i,5)=miss_internal
 endif

! calculate the standard deviation for ANNUAL
 if (frac_data) then
  total = 0
  do m=1,ioccur
   total = total + ((arr(m) - mean)*(arr(m) - mean))
  enddo
  if(ioccur.gt.1)total = total / real(ioccur-1.0)
  sd = sqrt(total)
  ts_sd(i,5)=sd
 else
  ts_sd(i,5)=miss_internal
 endif
enddo

! write the results for the mean
indexID=indexID+1
names(indexID)= 'wet_spell_mean'
if(newnames)names(indexID)= 'pwsav'
do i=minyr,maxyr
 do j=1,5
  ts(indexID,i,j)=ts_mean(i,j)
 enddo
enddo

! write the results for the 50th percentile
indexID=indexID+1
names(indexID)= 'wet_spell_perc'
if(newnames)names(indexID)= 'pwsmed'
do i=minyr,maxyr
 do j=1,5
  ts(indexID,i,j)=ts_50perc(i,j)
 enddo
enddo

! write the results for the standard deviation
indexID=indexID+1
names(indexID)= 'wet_spell_sd'
if(newnames)names(indexID)= 'pwssdv'
do i=minyr,maxyr
 do j=1,5
  ts(indexID,i,j)=ts_sd(i,j)
 enddo
enddo

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

! stats of dry spell lengths (days)

subroutine dry_spell_stats()
include 'indices.inc'
real p,quickpile,percent,mean,sd,total
integer i,j,k,m
integer crd,ioccur
integer realdaysinmth
real arr(nyrs*100)
real ts_mean(minyr:maxyr,5)
real ts_50perc(minyr:maxyr,5)
real ts_sd(minyr:maxyr,5)

logical frac_data

percent = 50.0

do i=minyr,maxyr
 crd=0
 ioccur = 0

! calc for DJF
 if(i.eq.minyr)then
  frac_data = .false.
 elseif (((frac(i-1,12,1)+frac(i,1,1)+frac(i,2,1))/3.0) .ge. frac_thresh) then
  do k=1,realdaysinmth(i-1,12)
!test for a consecutive dry day
   if (data_in(i-1,12,k,1) <= wd_cutoff .and. data_in(i-1,12,k,1) .ne.miss_internal) then ! dry day
    crd=crd+1
   else
    if (crd > 0) then
     ioccur = ioccur + 1
     arr(ioccur) = crd
    endif
    crd = 0
   endif
  enddo

  do j=1,2
   do k=1,realdaysinmth(i,j)
!test for a consecutive dry day
    if (data_in(i,j,k,1) <= wd_cutoff .and. data_in(i,j,k,1) .ne.miss_internal) then ! dry day
     crd=crd+1
    else
     if (crd > 0) then
      ioccur = ioccur + 1
      arr(ioccur) = crd
     endif
     crd = 0
    endif
   enddo
  enddo
  if (crd > 0) then
    ioccur = ioccur + 1
    arr(ioccur) = crd
  endif
  frac_data = .true.
 else
  frac_data = .false.
 endif

! calculate the mean for DJF
 if (frac_data) then
  total = 0
  do m=1,ioccur
   total = total + arr(m)
  enddo
  mean=total
  if(ioccur.gt.1)mean = total / real(ioccur)
  ts_mean(i,1)=mean
 else
  ts_mean(i,1)=miss_internal
 endif

! calculate the 50th percentile for DJF
 if (frac_data) then
  p=quickpile(arr,ioccur,percent,miss_internal)
  ts_50perc(i,1)=p
 else
  ts_50perc(i,1)=miss_internal
 endif

! calculate the standard deviation for DJF
 if (frac_data) then
  total = 0
  do m=1,ioccur
   total = total + ((arr(m) - mean)*(arr(m) - mean))
  enddo
  if(ioccur.gt.1)total = total / real(ioccur-1.0)
  sd = sqrt(total)
  ts_sd(i,1)=sd
 else
  ts_sd(i,1)=miss_internal
 endif

 crd=0
 ioccur=0

! calc for MAM
 if (((frac(i,3,1)+frac(i,4,1)+frac(i,5,1))/3.0) .ge. frac_thresh) then
  do j=3,5
   do k=1,realdaysinmth(i,j)
    if (data_in(i,j,k,1) <= wd_cutoff .and. data_in(i,j,k,1) .ne.miss_internal) then ! dry day
     crd=crd+1
    else
     if (crd > 0) then
      ioccur = ioccur + 1
      arr(ioccur) = crd
     endif
     crd = 0

    endif
   enddo
  enddo
  if (crd > 0) then
    ioccur = ioccur + 1
    arr(ioccur) = crd
  endif
  frac_data = .true.
 else
  frac_data = .false.
 endif

! calculate the mean for MAM
 if (frac_data) then
  total = 0
  do m=1,ioccur
   total = total + arr(m)
  enddo
  mean=total
  if(ioccur.gt.1)mean = total / real(ioccur)
  ts_mean(i,2)=mean
 else
  ts_mean(i,2)=miss_internal
 endif

! calculate the 50th percentile for MAM
 if (frac_data) then
  p=quickpile(arr,ioccur,percent,miss_internal)
  ts_50perc(i,2)=p
 else
  ts_50perc(i,2)=miss_internal
 endif

! calculate the standard deviation for MAM
 if (frac_data) then
  total = 0
  do m=1,ioccur
   total = total + ((arr(m) - mean)*(arr(m) - mean))
  enddo
  if(ioccur.gt.1)total = total / real(ioccur-1.0)
  sd = sqrt(total)
  ts_sd(i,2)=sd
 else
  ts_sd(i,2)=miss_internal
 endif

 crd=0
 ioccur=0

! calc for JJA
 if (((frac(i,6,1)+frac(i,7,1)+frac(i,8,1))/3.0) .ge. frac_thresh) then
  do j=6,8
   do k=1,realdaysinmth(i,j)
    if (data_in(i,j,k,1) <= wd_cutoff .and. data_in(i,j,k,1) .ne.miss_internal) then ! dry day
     crd=crd+1
    else
     if (crd > 0) then
      ioccur = ioccur + 1
      arr(ioccur) = crd
     endif
     crd = 0
    endif
   enddo
  enddo
  if (crd > 0) then
    ioccur = ioccur + 1
    arr(ioccur) = crd
  endif
  frac_data = .true.
 else
  frac_data = .false.
 endif

! calculate the mean for JJA
 if (frac_data) then
  total = 0
  do m=1,ioccur
   total = total + arr(m)
  enddo
  if(ioccur.gt.1)mean = total / real(ioccur)
  ts_mean(i,3)=mean
 else
  ts_mean(i,3)=miss_internal
 endif

! calculate the 50th percentile for JJA
 if (frac_data) then
  p=quickpile(arr,ioccur,percent,miss_internal)
  ts_50perc(i,3)=p
 else
  ts_50perc(i,3)=miss_internal
 endif

! calculate the standard deviation for JJA
 if (frac_data) then
  total = 0
  do m=1,ioccur
   total = total + ((arr(m) - mean)*(arr(m) - mean))
  enddo
  if(ioccur.gt.1)total = total / real(ioccur-1.0)
  sd = sqrt(total)
  ts_sd(i,3)=sd
 else
  ts_sd(i,3)=miss_internal
 endif

 crd=0
 ioccur=0

! calc for SON
 if (((frac(i,9,1)+frac(i,10,1)+frac(i,11,1))/3.0) .ge. frac_thresh) then
  do j=9,11
   do k=1,realdaysinmth(i,j)
    if (data_in(i,j,k,1) <= wd_cutoff .and. data_in(i,j,k,1) .ne.miss_internal) then ! dry day
     crd=crd+1
    else
     if (crd > 0) then
      ioccur = ioccur + 1
      arr(ioccur) = crd
     endif
     crd = 0
    endif
   enddo
  enddo
  if (crd > 0) then
    ioccur = ioccur + 1
    arr(ioccur) = crd
  endif
  frac_data = .true.
 else
  frac_data = .false.
 endif

! calculate the mean for SON
 if (frac_data) then
  total = 0
  do m=1,ioccur
   total = total + arr(m)
  enddo
  mean=total
  if(ioccur.gt.1)mean = total / real(ioccur)
  ts_mean(i,4)=mean
 else
  ts_mean(i,4)=miss_internal
 endif

! calculate the 50th percentile for SON
 if (frac_data) then
  p=quickpile(arr,ioccur,percent,miss_internal)
  ts_50perc(i,4)=p
 else
  ts_50perc(i,4)=miss_internal
 endif

! calculate the standard deviation for SON
 if (frac_data) then
  total = 0
  do m=1,ioccur
   total = total + ((arr(m) - mean)*(arr(m) - mean))
  enddo
  if(ioccur.gt.1)total = total / real(ioccur-1.0)
  sd = sqrt(total)
  ts_sd(i,4)=sd
 else
  ts_sd(i,4)=miss_internal
 endif

 crd=0
 ioccur=0

enddo

! calc the annual
do i=minyr,maxyr
 ioccur=0
 crd=0
 if (frac(i,13,1) .ge. frac_thresh) then
  do j=1,12
   do k=1,realdaysinmth(i,j)
    if (data_in(i,j,k,1) <= wd_cutoff .and. data_in(i,j,k,1) .ne.miss_internal) then ! dry day
     crd=crd+1
    else
     if (crd > 0) then
      ioccur = ioccur + 1
      arr(ioccur) = crd
     endif
     crd = 0
    endif
   enddo
  enddo
  if (crd > 0) then
    ioccur = ioccur + 1
    arr(ioccur) = crd
  endif
  frac_data = .true.
 else
  frac_data = .false.
 endif

! calculate the mean for ANNUAL
 if (frac_data) then
  total = 0
  do m=1,ioccur
   total = total + arr(m)
  enddo
  mean=total
  if(ioccur.gt.1)mean = total / real(ioccur)
  ts_mean(i,5)=mean
 else
  ts_mean(i,5)=miss_internal
 endif

! calculate the 50th percentile for ANNUAL
 if (frac_data) then
  p=quickpile(arr,ioccur,percent,miss_internal)
  ts_50perc(i,5)=p
 else
  ts_50perc(i,5)=miss_internal
 endif

! calculate the standard deviation for ANNUAL

 if (frac_data) then
  total = 0
  do m=1,ioccur
   total = total + ((arr(m) - mean)*(arr(m) - mean))
  enddo
  if(ioccur.gt.1)total = total / real(ioccur-1.0)
  sd = sqrt(total)
  ts_sd(i,5)=sd
 else
  ts_sd(i,5)=miss_internal
 endif
enddo

! write the results for the mean
indexID=indexID+1
names(indexID)= 'dry_spell_mean'
if(newnames)names(indexID)= 'pdsav'
do i=minyr,maxyr
 do j=1,5
  ts(indexID,i,j)=ts_mean(i,j)
 enddo
enddo

! write the results for the 50th percentile
indexID=indexID+1
names(indexID)= 'dry_spell_perc'
if(newnames)names(indexID)= 'pdsmed'
do i=minyr,maxyr
 do j=1,5
  ts(indexID,i,j)=ts_50perc(i,j)
 enddo
enddo

! write the results for the standard deviation
indexID=indexID+1
names(indexID)= 'dry_spell_sd'
if(newnames)names(indexID)= 'pdssdv'
do i=minyr,maxyr
 do j=1,5
  ts(indexID,i,j)=ts_sd(i,j)
 enddo
enddo

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!end of precipitation stats subs !!!!!!!!!!!!!

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! Utilities
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

subroutine normalize(id,muse,ivar)

! normalize results for missing data

include 'indices.inc'

integer i,j,tnum,issn,muse(12),id,ivar
real tfrac

! normalize
do i=minyr,maxyr

 do issn=1,4
  if(ts(id,i,issn).ne.miss_internal)then !results here
   if(frac_ssn(i,issn,ivar).lt.frac_thresh)then
    ts(id,i,issn)=miss_internal
   else
    ts(id,i,issn)=ts(id,i,issn)/frac_ssn(i,issn,ivar)
   endif
  endif
 end do

 if(ts(id,i,5).ne.miss_internal)then !results here
  tnum=0
  tfrac=0
  do j=1,12
   if(muse(j).eq.1)then
    tfrac=tfrac+frac(i,j,ivar)
    tnum=tnum+1
   endif
  enddo
  if(tnum > 0)then
   tfrac=tfrac/tnum
  else
   tfrac=1	! haven't used any months so don't want tto set to missing
  endif
  if(tfrac.lt.frac_thresh)then
   ts(id,i,5)=miss_internal
  else
   ts(id,i,5)=ts(id,i,5)/tfrac
  endif
 endif

enddo

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

  subroutine fraction

! this program tests to see how many data points are in the months
! in which this index may be calculated

! current plans are (a) must have >/= 90% data & then normalize by the
! fraction of data present:  e.g., if 15 days > 25C & month has 5% missing
! the answer would be 15x1.05=15.75

include 'indices.inc'
integer i,j,k,l
integer kmax
real rnum,v
integer realdaysinmth
integer yr,ivar,issn,mth,iyr,imth

! monthly
do i=minyr,maxyr
 do j=1,12
  kmax=realdaysinmth(i,j)
  do l=1,4
   rnum=0
   do k=1,kmax
    if(data_in(i,j,k,l).ne.miss_internal)rnum=rnum+1
   enddo
   frac(i,j,l)=rnum/real(kmax)
  enddo
 enddo
enddo

! annual
do i=minyr,maxyr
 do l=1,4
  v=0
  do j=1,12
   v=v+frac(i,j,l)
  enddo
  v=v/12
  frac(i,13,l)=v
 enddo
enddo

! seasonal
do yr=minyr,maxyr
 do ivar=1,4
  do issn=1,4
   v=0
   do mth=1,3
    if(issn.eq.1.and.mth.eq.1)then
     iyr=yr-1
    else
     iyr=yr
    end if
    imth=(issn-1)*3+mth-1
    if(imth.eq.0)imth=12
    if(iyr.ge.minyr)v=v+frac(iyr,imth,ivar)/3.0
   end do
   frac_ssn(yr,issn,ivar)=v
  end do
 end do
end do
   
return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

subroutine normals()

! This subroutine takes the main data array 'data' from ClimateIndices.f
! and calculates the start year to end year daily normals using the 5-day
! window technique.

include 'indices.inc'

integer i,y1,m1,d1,y2,m2,d2,ioffset
real arr(bm_minyr:bm_maxyr,-2:2)
integer daysinmth,ssn(12),issn,iday,d,date_compare,mth,realdaysinmth

! ignore leap years - normals for Feb 29th set to average of Feb 28th and March 1st

! init
do i=1,4
 do m1=1,12
  do d1=1,31
   DailyNormals(m1,d1,i)=miss_internal
  end do
 end do
 do m1=1,4
  do d1=1,93
   DailyNormals_ssn(m1,d1,i)=miss_internal
  end do
 end do
end do

do i=1,4
 do m1=1,12
  do d1=1,daysinmth(m1)
   do y1=bm_minyr,bm_maxyr
    do ioffset=-2,2
     call date_offset(y1,m1,d1,y2,m2,d2,ioffset)
     if(date_compare(y2,m2,d2,bm_minyr,1,1).ne.-1.and.date_compare(y2,m2,d2,bm_maxyr,12,31).ne.1)then
      arr(y1,ioffset)=data_in(y2,m2,d2,i)
     else
      arr(y1,ioffset)=miss_internal
     end if
    end do
   end do
   call average((bm_maxyr-bm_minyr+1)*5,miss_internal,arr,DailyNormals(m1,d1,i))
  enddo
 enddo
 DailyNormals(2,29,i)=(DailyNormals(2,28,i)+DailyNormals(3,1,i))/2.0
enddo

! seasonal normals (same as daily)
ssn(1)=1
ssn(2)=1
ssn(3)=2
ssn(4)=2
ssn(5)=2
ssn(6)=3
ssn(7)=3
ssn(8)=3
ssn(9)=4
ssn(10)=4
ssn(11)=4
ssn(12)=1
do i=1,4
 issn=1
 iday=0
 do d=1,31
  iday=iday+1
  DailyNormals_ssn(issn,iday,i)=DailyNormals(12,d,i)
 end do
 do mth=1,11
  if(ssn(mth).ne.issn)iday=0
  issn=ssn(mth)
  do d=1,realdaysinmth(1996,mth)
   iday=iday+1
   DailyNormals_ssn(issn,iday,i)=DailyNormals(mth,d,i)
  end do
 end do
end do

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

subroutine normals_perc(perc)

! This subroutine takes the main data array 'data' from ClimateIndices.f
! and calculates the start year to end year daily normals (percentile) using the 5-day
! window technique.

include 'indices.inc'

integer i,y1,m1,d1,y2,m2,d2,ioffset
real arr(bm_minyr:bm_maxyr,-2:2)
real perc,quickpile
integer daysinmth,ssn(12),issn,iday,d,date_compare,mth,realdaysinmth

! ignore leap years - normals for Feb 29th set to average of Feb 28th and March 1st

! init
do i=1,4
 do m1=1,12
  do d1=1,31
   DailyNormals_perc(m1,d1,i)=miss_internal
  end do
 end do
 do m1=1,4
  do d1=1,93
   DailyNormals_ssn_perc(m1,d1,i)=miss_internal
  end do
 end do
end do

do i=1,4
 do m1=1,12
  do d1=1,daysinmth(m1)
   do y1=bm_minyr,bm_maxyr
    do ioffset=-2,2
     call date_offset(y1,m1,d1,y2,m2,d2,ioffset)
     if(date_compare(y2,m2,d2,bm_minyr,1,1).ne.-1.and.date_compare(y2,m2,d2,bm_maxyr,12,31).ne.1)then
      arr(y1,ioffset)=data_in(y2,m2,d2,i)
     else
      arr(y1,ioffset)=miss_internal
     end if
    end do
   end do
   DailyNormals_perc(m1,d1,i)=quickpile(arr,(bm_maxyr-bm_minyr+1)*5,perc,miss_internal)
  enddo
 enddo
 DailyNormals_perc(2,29,i)=(DailyNormals_perc(2,28,i)+DailyNormals_perc(3,1,i))/2.0
enddo

! seasonal normals (same as daily)
ssn(1)=1
ssn(2)=1
ssn(3)=2
ssn(4)=2
ssn(5)=2
ssn(6)=3
ssn(7)=3
ssn(8)=3
ssn(9)=4
ssn(10)=4
ssn(11)=4
ssn(12)=1
do i=1,4
 issn=1
 iday=0
 do d=1,31
  iday=iday+1
  DailyNormals_ssn_perc(issn,iday,i)=DailyNormals_perc(12,d,i)
 end do
 do mth=1,11
  if(ssn(mth).ne.issn)iday=0
  issn=ssn(mth)
  do d=1,realdaysinmth(1996,mth)
   iday=iday+1
   DailyNormals_ssn_perc(issn,iday,i)=DailyNormals_perc(mth,d,i)
  end do
 end do
end do

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

subroutine average(n,miss,rData,rAve)

real rData(n),miss

rCount=0
rSum = 0.0

do i=1,n
  if (rData(i).ne.miss) then
    rCount=rCount+1.
    rSum = rSum+rData(i)
  endif
enddo

if (rCount.gt.0.) then
  rAve=(rSum/rCount)
else
  rAve=miss
endif

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

real function quickpile(x,nvar,pctile,miss1)
! returns "pctile" percentile
! uses 'Empirical Distribution Function' method with interpolation

include 'indices.inc'

integer nvar,i
real x(nvar),pctile,miss1,rtile,f,sorted(nvar)
integer nonmiss

if(pctile.lt.0.0.or.pctile.gt.100.0)then
 stop 'quickpile: invalid percentile'
 return
end if

! sort
call quicksort(x,nvar,sorted,miss1,nonmiss)

! get percentile
if(nonmiss.lt.minPercentileDays)then
 quickpile=miss1
else
 rtile=(nonmiss-1)*pctile/100.0
 i=int(rtile)
 f=rtile-real(i)
 quickpile=sorted(i+1)
 if(f.gt.0.00001)then
  quickpile=quickpile+f*(sorted(i+2)-sorted(i+1))
 end if
end if

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

integer function realdaysinmth(yr,mth)
! return the real number of days in a month given the year

implicit none

integer yr,mth,daysinmth
logical leapYr

if(mth.eq.2)then
 if(leapYr(yr))then
  realdaysinmth=29
 else
  realdaysinmth=28
 end if
else
 realdaysinmth=daysinmth(mth)
end if
return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

LOGICAL FUNCTION  leapYr(nYr)
! This function   returns  logical true if the given year is a      *
! leap year.                                                        *
!
! LEAP YEAR RULE:                                                   *
! 1/. If the year is evenly divisible by 400 then it is a leap year.*
!
! 2/. Century years, except those under rule 1, are NOT leap years. *
!
! 3/. All non-century years evenly divisible by 4 are leap years.   *
!************************************************************************
INTEGER nYr

if (MOD(nYr,400) .eq. 0) then
   leapYr = .true.
else
   if (MOD(nYr,100).eq. 0) then
      leapYr = .false.
   else
      if (MOD(nYr,4).eq. 0) then
         leapYr = .true.
      else
         leapYr = .false.
      end if
   end if
end if
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

subroutine quickrank_miss(n,arrin,indx,miss)
! rank a series that has missing values

implicit none

integer n,indx(n),i,indx1(n)
real arrin(n),miss
integer pos(n),nonmiss
real data(n)

nonmiss=0
do i=1,n
 if(arrin(i).ne.miss)then
  nonmiss=nonmiss+1
  data(nonmiss)=arrin(i)
  pos(nonmiss)=i
 endif
end do
if(nonmiss.gt.0)call quickrank(nonmiss,data,indx1)

do i=1,n
 if(i.le.nonmiss)then
  indx(i)=pos(indx1(i))
 else
  indx(i)=0
 end if
end do

end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

subroutine quickrank(n,arrin,indx)
! rank an array
! indx(1) is greatest value

implicit none

integer n,l,indx(n),i,j,ir,indxt
real arrin(n),q

do j=1,n
 indx(j)=j
end do
if(n.eq.1)return
l=n/2+1
ir=n
 10 continue
if(l.gt.1)then
 l=l-1
 indxt=indx(l)
 q=arrin(indxt)
else
 indxt=indx(ir)
 q=arrin(indxt)
 indx(ir)=indx(1)
 ir=ir-1
 if(ir.eq.1)then
  indx(1)=indxt
! reverse order
  j=0
 30   continue
  j=j+1
  if(j.ge.n+1-j)return
  i=indx(j)
  indx(j)=indx(n+1-j)
  indx(n+1-j)=i
  goto 30
 end if
end if
i=l
j=l+l
 20 continue
if(j.le.ir)then
 if(j.lt.ir)then
  if(arrin(indx(j)).lt.arrin(indx(j+1)))j=j+1
 end if
 if(q.lt.arrin(indx(j)))then
  indx(i)=indx(j)
  i=j
  j=j+j
 else
  j=ir+1
 end if
 goto 20
end if
indx(i)=indxt
goto 10
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

integer function daysinmth(mth)
! function to return the number of days in a month (FEB=28)

! ***INPUT***
! mth [integer]=month number
! ***OUPUT***
! daysinmth [integer] = no. of days in month mth

integer mth
if(mth.lt.1.or.mth.gt.12)stop 'Illegal mth in funcn daysinmth'
if(mth.eq.1)then
  daysinmth=31
else if(mth.eq.2)then
  daysinmth=28
else if(mth.eq.3)then
  daysinmth=31
else if(mth.eq.4)then
  daysinmth=30
else if(mth.eq.5)then
  daysinmth=31
else if(mth.eq.6)then
  daysinmth=30
else if(mth.eq.7)then
  daysinmth=31
else if(mth.eq.8)then
  daysinmth=31
else if(mth.eq.9)then
  daysinmth=30
else if(mth.eq.10)then
  daysinmth=31
else if(mth.eq.11)then
  daysinmth=30
else if(mth.eq.12)then
  daysinmth=31
end if
return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

integer function daysinssn(ssn)
! function to return the number of days in a season (FEB=28)

implicit none

! ***INPUT***
! ssn [integer]=season number (1=DJF ... 5=ANN)
! ***OUPUT***
! daysinssn [integer] = no. of days in season

integer ssn,daysinmth
integer mth(4,3),j

mth(1,1)=12
mth(1,2)=1
mth(1,3)=2
mth(2,1)=3
mth(2,2)=4
mth(2,3)=5
mth(3,1)=6
mth(3,2)=7
mth(3,3)=8
mth(4,1)=9
mth(4,2)=10
mth(4,3)=11

if(ssn.lt.1.or.ssn.gt.5)stop 'Illegal ssn in funcn daysinssn'

if(ssn.eq.5)then
 daysinssn=365
else
 daysinssn=0
 do j=1,3
  daysinssn=daysinssn+daysinmth(mth(ssn,j))
 end do
end if
return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

integer function realdaysinssn(yr,ssn)
! return the real number of days in a season given the year

implicit none

integer yr,ssn,daysinssn
logical leapYr

realdaysinssn=daysinssn(ssn)
if(leapYr(yr).and.(ssn.eq.1.or.ssn.eq.5))realdaysinssn=realdaysinssn+1
return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

subroutine quicksort(x,n,y,miss,nonmiss)
! sort array x into ascending order into y
! n returns the number of non missing values

implicit none

integer n,i,indx(n),nonmiss
real x(n),y(n),miss

call quickrank_miss(n,x,indx,miss)

! count non missing
nonmiss=0
do i=1,n
 if(x(i).ne.miss)nonmiss=nonmiss+1
 y(i)=miss
end do

do i=1,nonmiss
 y(i)=x(indx(nonmiss-i+1))
end do

return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

subroutine date_offset(yr,mth,day,yr1,mth1,day1,day_offset)
! calculate date with day offset

implicit none

integer yr,mth,day,yr1,mth1,day1,day_offset,realdaysinmth

yr1=yr
mth1=mth
day1=day+day_offset

10 continue

if(day1.gt.realdaysinmth(yr1,mth1))then
 day1=day1-realdaysinmth(yr1,mth1)
 mth1=mth1+1
 if(mth1.gt.12)then
  mth1=1
  yr1=yr1+1
 end if
 goto 10
end if
 
if(day1.lt.1)then
 mth1=mth1-1
 if(mth1.lt.1)then
  mth1=12
  yr1=yr1-1
 end if
 day1=day1+realdaysinmth(yr1,mth1)
 goto 10
end if
 
return
end

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

integer function date_compare(yr,mth,day,yr1,mth1,day1)
! compare dates
! -1: yr/mth/day is before yr1,mth1,day1
!  0: same
!  1: yr/mth/day is after yr1,mth1,day1

implicit none

integer result,yr,mth,day,yr1,mth1,day1

if(yr.gt.yr1)then
 result=1
elseif(yr.lt.yr1)then
 result=-1
else
 if(mth.gt.mth1)then
  result=1
 elseif(mth.lt.mth1)then
  result=-1
 else
  if(day.gt.day1)then
   result=1
  elseif(day.lt.day1)then
   result=-1
  else
   result=0
  end if
 end if
end if

date_compare=result
return
end
